globals [
  #I           ;; # of Infectious     - red
  #R           ;; # of Removeds       - green   (note: not exactly the integral of #I)
  nI           ;; fractional I for the fitted Normal curve
  nR           ;; fractional R for the fitted Normal curve
  rR0          ;; to align Normal-R with Reach-R at liftoff
  RedsQ        ;; list of red patches - to remove the oldest first
  T#           ;; # of transmissible (i.e. always infects susceptibles) contacts per household
  meanReach    ;; mean infection distance  = radius that spans meanReach% of HHs
  gamma        ;; the removal rate per day for infectious households
  N            ;; # households  (that fill the whole World)
  #Imax        ;; peak #I
  ImaxDay      ;; peak #I day
  Istdev       ;; standard deviation (width of the "bell") for a Normal curve that best fits the Proximity model I-curve
  Ilist        ;; list of daily #Is - used to calc VAR of the fitted I curve
  plottingReach;; a switch to turn this off when plotting over it
  liftoff      ;; switch = (#I / N > .001)
  day          ;; for plotting Normal curves
  stopGos
]

to runCurrentT#  ;; avoid having to setup first
  setup
  while [not stopGos] [go]  ;; run for current T#
end

to runNextT#     ;; to dynamically show the effect of stepping up through each T#
  set Transmission#  precision (Transmission# + 0.1) 5   ;; Increment the T#  (note: precision is to correct an anomylous addition error)
  runCurrentT#
end

to setup
  clear-all
  set stopGos   false       ;; enable resumption of go's for this Transmission# - used in runCurrentTransmission#

  ;; setup the Reach model
  set N          count patches
  set T#         Transmission#                ;; the T# abbreviation will be used throughout this model
  set gamma      1 / generation-perHousehold  ;; removeds per day per infectious HH
  set meanReach  sqrt(meanReach% / 100 * N / pi)
  set RedsQ      []
  ask patch      (world-width / 2) (world-width / 2) [turn-infectious]            ;; infect the origin
  set #I         1
  set #R         0
  set #Imax      #I
  set Ilist      []

  ;; LIFTOFF - startup until liftoff (when #I >= N * .001).  During liftoff "infect&remove" avoids accidentally exhausting all infections.
  set liftoff  false                                                              ;; start liftoff
  while [#I < N * .001] [goReachModel]                                            ;; run till liftoff
  set liftoff  true                                                               ;; achieved liftoff
  set rR0  #R / N                                                                 ;; to align Normal-R with Reach-R at liftoff in plotNORMdistFit
  set plottingReach true                                                          ;; enable plot of Reach curves until we plot Normal curves over the top of them

  reset-ticks
end

to go
  goReachModel
  ;; tally Imax and ImaxDay
  if #I > #Imax  [
    set #Imax    #I
    set ImaxDay  ticks + 1
  ]

  ;; stop when I is passed its peak and nearly exhausted
  if (ticks > ImaxDay) and (#I <= N * .0001) [
    print (word "N=" N  ", genDays=" generation-perHousehold ", Reach=" meanReach%  "%, T#=" PadRight T# 3  ", ImaxDay=" PadRight ImaxDay 4 ", %Imax=" precision (#Imax / N * 100) 2 ", %R=" precision (#R / N * 100) 1)
    plotNORMdistFit
    set stopGos true                                                              ;; tell runNextTransmission# to stop go's
    stop
  ]

  tick                                                                            ;; advance to next day
end

to goReachModel
  ;; step the proximity model today. Transmit then remove the oldest reds, according to the removal rate, gamma
  set Ilist  lput #I  Ilist                                                       ;; append #I to Ilist
  if #I = 0 [stop]
  let #newR  max list 1 round(gamma * #I)                                         ;; the number of oldest reds - to transmit and then remove
  foreach   (sublist RedsQ 0     #newR         ) [HH -> ask HH [infect&remove]]  ;; transmit these oldest reds (those at the front of the queue), which adds new reds to the back of the queue
  set RedsQ (sublist RedsQ #newR (length RedsQ))                                  ;; drop these oldest reds from the front of the list, now that they have been processed
end

to infect&remove  ;; infect and then remove
  let #Contacts  ifelse-value liftoff [random-poisson     T#       ] [1 + random-poisson     (T# - 1)       ]
  let Reach      ifelse-value liftoff [random-exponential meanReach] [1 + random-exponential (meanReach - 1)]

  repeat #Contacts [ ;; make a transmissible contact at distance random-exponential of meanReach
    ask patch-at-heading-and-distance (random 360) Reach [if pcolor = black [turn-infectious]]
  ]
  turn-removed                                                                    ;; remove this HH, now that it has transmitted
end

to turn-infectious    ;; turn into an "infectious"
  set RedsQ  lput self RedsQ                                                      ;; put new red at the back of the queue
  set pcolor red      ;; set infectious
  set #I  #I + 1
end

to turn-removed       ;; turn into a "removed"
  set pcolor green    ;; set removed
  set #I  #I - 1
  set #R  #R + 1
end

to-report PadRight [number to-length]  ;; to align decimal columns in output
  report substring (word number "             ") 0 to-length
end

to plotNORMdistFit
  ;; plot nI (the Normal curve that fits the Reach I curve) and nR (the R curve that is the accumulation of nI * gamma), alongside the Reach I and R curves

  set plottingReach false   ;; to cease plotting the Reach curves while plotting the Normal curves over the Reach curves

  ;; plot the nI and nR plots
  let Imax    #Imax / N
  let cumI    #R / gamma                                                          ;; cumI = the integral of the Reach #I-curve
  set Istdev  cumI / (#Imax * sqrt(2 * pi))

  set nR   rR0                                                                    ;; to align nR with rR at liftoff

  set day  (- 1)
  set day  0
  repeat ticks + 1 [
    set day  day + 1
    set nI   Imax * exp(-(((day - ImaxDay)/ Istdev)^ 2)/ 2)                       ;; the Normal distribution with this ImaxDay and Istdev
    set nR   nR + nI * gamma
    update-plots
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
257
10
765
519
-1
-1
1.0
1
10
1
1
1
0
1
1
1
0
499
0
499
1
1
1
tick
30.0

BUTTON
189
10
247
43
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
189
47
246
80
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

PLOT
0
375
926
891
Reach model (with a fitted Normal-distribution curve)
Day
%
0.0
180.0
0.0
60.0
true
true
"" ""
PENS
"reach Infectious%" 1.0 0 -2674135 true "" "if plottingReach [plot #I / N * 100]"
"reach Removed%" 1.0 0 -10899396 true "" "if plottingReach [plot #R / N * 100]"
"normal Infectious%" 1.0 0 -16777216 true "" "plotxy day  nI * 100"
"normal Removed%" 1.0 0 -16777216 true "" "plotxy day  nR * 100"

BUTTON
189
84
246
117
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
83
256
152
301
Imax%
#Imax / N * 100
2
1
11

SLIDER
0
79
181
112
Transmission#
Transmission#
1.3
5
1.6
.1
1
NIL
HORIZONTAL

SLIDER
0
10
182
43
generation-perHousehold
generation-perHousehold
4
7
5.0
1
1
days
HORIZONTAL

MONITOR
84
199
151
244
NIL
#I
17
1
11

MONITOR
15
255
78
300
NIL
ImaxDay
17
1
11

MONITOR
158
199
227
244
NIL
#R
17
1
11

SLIDER
0
45
181
78
meanReach%
meanReach%
.2
2
0.4
.2
1
NIL
HORIZONTAL

MONITOR
0
199
79
244
NIL
N
17
1
11

MONITOR
159
256
226
301
R%
#R / N * 100
1
1
11

BUTTON
69
120
246
153
run current Transmission#
runCurrentT#
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
69
160
246
193
run the next Transmission#
runNextT#
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
## WHAT IS IT?

This model simulates the spread of an infectious disease in a closed population. 

It diverges from the classical Kermack-McKendrick model, which we henceforth call the **Standard model**, in that it takes account of **spatial proximity** on transmission. It is novel in that it doesn't model the movement of persons but instead assumes that transmission is between static households, and that households can transmit at a distance.  Of course in reality, the way this remote transmission takes place is by the movement of persons from the households.  However, we don't need to model this movement.  We simply assume that transmission is between static households and that it is more likely to be between nearer households. For simplicity, every patch is a household. 

This **proximity effect** is determined by a Reach parameter which is the average distance of transmission to other households.  For example, a Reach of 1% means that transmissions are to households that are at a 1% radius of the transmitter, on average. The households at 1% radius are those on the boundary of the 1% of all households that are closest to the transmitter.  Note that this 1% radius is only an average. The actual transmission radius is a random-Poisson of the 1% radius.  

This is a simple model that is designed only to show the Reach-effect on the spread of infection. It is for experimental purposes only, so that users can see how the **Reach%** compresses and depresses the spread of the infection and how it differs significantly from the non-spatial Standard model.

I have avoided using the standard term R_0 as it is often misused and could be confused with some of my terms relating to the Removeds measure.  Instead I use the term **Transmission#** (also abbreviated to T#) as I think this is much more self-descriptive.

There are 3 main parameters that can be experimented with using the sliders:
  * generation-perHousehold
  * meanReach%, and
  * Transmission#

The first, **generation-perHousehold**, requires a little explanation.  It means the average number of days from first infection in a household till the day it reinfects other households.  This is complicated as there will sometimes be a delay from the first infection in a household and any reinfection of other members in the household who will in turn elongate the total reinfection interval of the household.  I have assumed a default which is my guess for Covid-19 and have assumed that secondary infections within the household have little effect on the household reinfection interval. In terms of the Standard Model it equals 1/gamma.

## HOW IT WORKS

Initially, the household at the centre of the World is infected.  This household transmits an average T# times at the end of its reinfection interval to other nearby households and then becomes Removed (i.e. recovered or deceased).  Infected households, in turn do the same.  Thus all households transmit at the end of their reinfection interval, making the model temporally-aware as well as spatially-aware. Temporal awareness only makes a difference because the model is spatially-aware. Restricting transmission to the end of the reinfection interval ensures that the oldest (and hence the less-distributed) households transmit each day.  This transmission by the oldest infected is implemented using a queueing system for infected households.

The state of each household is shown by its colour:
  * red for the         Infectious
  * green for the       Removed 
  * black for the still Susceptible

The %Infectious and the %Removed is plotted by day. Examination of the %Infectious plot shows it is approximately the shape of a Normal-distribution curve. It is therefore overlaid with a fitted Normal-distribution curve and the %Removed curve is overlaid with the corresponding fitted curve that is gamma * (the daily accumulated fitted %Infectious). You can see that they are a very good fit. 

The reason why the Normal-distribution curve is a better fit than the Standard-model curves is because infection is suppressed by the Reach-effect and the Standard-model does not take this into account. Also, the Standard-model %Infectious curves show a positive-skew (increasing markedly with T# above 4).  Whereas, the Reach-model shows very little skew in the Reach% range of 0.6% to 1.4%, a slight negative-skew when Reach% < 0.6% and a slight positive-skew when Reach% > 1.4%. I suspect that this could be corrected by introducing a skew coefficient into the Random-distribution formula.

## HOW TO USE IT

First, run it with the "view updates" on and observe how the red and green household dots spread through the World, eventually leaving just completely-randomised green households. However, this slows down the model and you will soon want to speed up the runs and plotting by unclicking "view updates" in the Interface ribbon. 

As per usual, you can adjust the input-parameter sliders then click the Setup button and then click the Go-forever button (or you can click the Go botton to run single daily steps). 

However, you can just click the "run current Transmission#" button to do both. Alternatively, you can click the "run the next Transmission#" button to step-up to the next Transmission# and automatically run that.  This latter button is a great-way of observing dynamically how the increasing Transmission# affects the plots.

You can observe how good the Normal-curve fit is. This fit will vary on repeated runs due to stochastic noise and occasionally you will observe an almost perfect fit.

The other item that you may want to vary is the number of households.  The default is 250,000 for a World that is 500 by 500.  The only way to change this is to resize the World which you can do by clicking Settings on the Interface ribbon.For example, you can set a World that is 1000 by 1000 to produce a number of households (N) of 1 million.  I have tested this up to 20 million households, which equates to an approximate population of 50 million persons.  Beware though that changing to a bigger World size can be slow and it slows down the run time considerably.

## HOW TO CITE

If you mention this model or the NetLogo software in a publication, we ask that you include the citations below.

For the model itself:

* Ian J Heath, the author, an independent researcher in Canterbury, UK

For the NetLogo software:

* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

person lefty
false
0
Circle -7500403 true true 170 5 80
Polygon -7500403 true true 165 90 180 195 150 285 165 300 195 300 210 225 225 300 255 300 270 285 240 195 255 90
Rectangle -7500403 true true 187 79 232 94
Polygon -7500403 true true 255 90 300 150 285 180 225 105
Polygon -7500403 true true 165 90 120 150 135 180 195 105

person righty
false
0
Circle -7500403 true true 50 5 80
Polygon -7500403 true true 45 90 60 195 30 285 45 300 75 300 90 225 105 300 135 300 150 285 120 195 135 90
Rectangle -7500403 true true 67 79 112 94
Polygon -7500403 true true 135 90 180 150 165 180 105 105
Polygon -7500403 true true 45 90 0 150 15 180 75 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Record parameters to fit the I-curve" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <steppedValueSet variable="meanReach%" first="0.2" step="0.2" last="1.4"/>
    <steppedValueSet variable="C" first="1.5" step="0.1" last="3"/>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
