turtles-own [
  fitness        ;; equal to the solution value of the objective function at the turtle's current location
]

patches-own [
  solution       ;; the solution (energy or fitness) for the x,y values (may be negative)
]

globals [
  winner         ;; not really necessary since there is only one turtle at a time, but it is the turtle 
                 ;; that currently has the best solution
  total-work     ;; amount of work that is has been done. Essentially the number of times a solution value
                 ;; has been calculated
  best-fitness   ;; the current best solution - again, not really "best" since there is only one current solution 
  global-best-fitness  ;; the best solution that has ever been found during the progress
  global-best-loc      ;; where the best solution was found
  g-x   ;; temporary variables that hold the best x and y location
  g-y
  
  temperature   ;; the current temperature of the system. Starts at 100 and the algorithm terminates when it gets below 1.

  max-solution  ;; the greatest patch value prior to scaling. used during setup.
  top-solution  ;; the ultimate greatest patch value after scaling.
  low-solution  ;; the lowest patch value prior to scaling. used during setup.
]

;==========SETUP=============
to setup
  clear-all
  
;; create the 2D domain of solutions
  set max-solution -100
  set top-solution 400
  create-solutions

  set total-work 0 ;; we start off at 0 work performed, obviously.

;; create the turtle that will represent the algorithm's current parameter set (i.e., location).
  create-turtles 1 [
     setxy random-xcor random-ycor ;; start the turtle at a random solution point
     set color cyan
     set shape "circle"
     ifelse trace-on? [ pen-down ] [ pen-up ]
     calculate-fitness
     set total-work (total-work + 1) ;; everytime we calculate a solution, we add 1 to our total work counter.
  ]

;; populate the variables with "best" solution information
  set winner max-one-of turtles [fitness]
  set best-fitness [fitness] of winner
  set global-best-fitness best-fitness
  set g-x [xcor] of winner
  set g-y [ycor] of winner
  set global-best-loc (list g-x g-y)
  
;; set the initial temperature of the system
  set temperature 100

  reset-ticks
end
;=====================================

;===========GO========================
to go
;; anneal at this temperature
  anneal-turtle

;; populate the variables with "best" solution information
  set winner max-one-of turtles [fitness]
  set best-fitness [fitness] of winner
  if best-fitness > global-best-fitness [ 
      set global-best-fitness best-fitness 
      set g-x [xcor] of winner
      set g-y [ycor] of winner
      set global-best-loc (list g-x g-y)
     ]

;; reduce the temperature based on user input
  set temperature temperature * (1 - cooling-rate / 100)
  
;; determine if we stop. If so, put the overall best solution in the monitors.
  if (temperature < 1)
    [ ask turtles 
      [
      set color yellow
      setxy (item 0  global-best-loc) (item 1  global-best-loc)
      ]
      stop ]
  tick
end
;=====================================

;===========CREATE-SOLUTIONS==========
to create-solutions
;; solutions are stored as the "solution" patch variable
  ask patches [
;; for each patch, we initially set the solution to zero, just so something is in the variable at each location.
    set solution 0
;; we will now put a bunch of "single spires" in the patch. We will eventually smooth these individual spires to make them "humps".
      if random-float 1 < .5 
       [ ;; controls the number of spires, on a per patch basis - so want a low probability
          ifelse random-float 1 < 0.25 
             [set solution -.25]
             [set solution 1 ]
          set solution (solution * random (top-solution * 100))
       ]
  ]
;; smooth the spires to make humps    
     repeat 100 [ diffuse solution 1.0 ]
  
  
;; now we will add a bit of small scale variability to the solution space - i.e., bumps.
  ask patches [
    set solution ( solution + random (top-solution / 20)) 
  ]
    
;;Now for the scaling of solution
;; adjust all solutions to a height proportional to the overall solution of the patch
  set max-solution max [solution] of patches
  if max-solution > 0 
    [ ask patches 
      [ set solution ((solution   /  max-solution  ) * top-solution ) ]
    ]
  set max-solution max [solution] of patches
  set low-solution min [solution] of patches
  
;; now we will color the patches to make a nice visualization  
  ask patches [
     set pcolor scale-color brown solution low-solution top-solution 
  ]
  
;; let's highlight the best solution (maximum patch) by coloring it red  
  let best-patch max-one-of patches [solution]
  ask best-patch [ set pcolor red ]
  
end
;=====================================

;=======CALCULATE-FITNESS=============
to calculate-fitness       
;; a turtle procedure that returns the patch solution where the turtle is
  set fitness [solution] of patch-here
end
;=====================================

;=======ACCEPT-CHANGE?================
to-report accept-change? [ old-energy new-energy ]
;; a reporter that will return true or false to indicate whether the turtle will move (accept)
;; the new solution location, or stay where it is.

  report (new-energy > old-energy)       ;; always accept new location if better.
         or (accept-equal-changes? and new-energy = old-energy) ;; accept new location at equal value if user control says so
;; the following line is the key simulated annealing control. The idea is that as the temperature is reduced, it is less likely
;; to move to a poorer new location. When the temperature is high, the probabily of moving to poorer locations is greater.
;; this follows the Boltzmann Distribution
         or ( exp ((old-energy - new-energy) * -1 / (0.1 * temperature)) > random-float 1.0 )
end
;=====================================

;=======ANNEAL-TURTLE================
to anneal-turtle
;; figure out what the new potential solution is, and determine whether to move there or stay put.
  ask turtles [ ;; there is only one turtle...
;; in this 2D example, a new solution can be found by going a distance in a direction. We will use
;; the built-in turtle moving routines to make this easy to program.
;; pick a random direction
    right random 360
;; get a random distance that is limited by the user control
    let my-distance max-pxcor  * delta-max / 100 * random-float 1.

;; figure out what the solution is for the new distance and compare to current solution
    if ( can-move? my-distance )
       [ let o-energy fitness
         let n-energy [solution] of patch-ahead my-distance 
         set total-work (total-work + 1)
         
         if (accept-change? o-energy n-energy )
;; we have determined to move to the new solution, so do it!
           [ 
             move-to patch-ahead my-distance
             calculate-fitness
;; we don't increment the work counter, since we really already calculated it and accounted for it just previously.
           ] 
       ]
  ]
end


; Portions COpyright 2012 Kevin Brewer. All rights reserved.
; Portions Copyright 2008 Uri Wilensky. All rights reserved.
; The full copyright notice is in the Information tab.
@#$#@#$#@
GRAPHICS-WINDOW
545
17
1158
651
100
100
3.0
1
10
1
1
1
0
1
1
1
-100
100
-100
100
1
1
1
ticks
30.0

BUTTON
108
108
193
141
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
20
68
193
101
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
20
148
192
181
cooling-rate
cooling-rate
.1
10
0.5
.1
1
NIL
HORIZONTAL

PLOT
200
68
530
218
Solution Progress
step
Solution Value
0.0
20.0
0.0
10.0
true
false
"" ""
PENS
"best" 1.0 0 -2674135 true "" "plot best-fitness"

BUTTON
20
108
105
141
step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
361
224
434
269
Total Work
total-work
0
1
11

MONITOR
216
277
334
322
Best Solution Value
global-best-fitness
0
1
11

MONITOR
362
277
525
322
Best Solution Parameters (x,y)
global-best-loc
0
1
11

MONITOR
204
225
335
270
Current Temperature
temperature
5
1
11

SWITCH
10
185
194
218
accept-equal-changes?
accept-equal-changes?
0
1
-1000

SLIDER
19
223
191
256
delta-max
delta-max
1
50
20
1
1
%
HORIZONTAL

SWITCH
49
261
158
294
trace-on?
trace-on?
0
1
-1000

@#$#@#$#@
## WHAT IS IT?

This model demonstrates the use of a simulated annealing algorithm on a very simple two-dimensional problem.  Simulated annealing is an optimization technique inspired by the natural annealing process used in metallurgy, whereby a material is carefully heated or cooled to create larger and more uniform crystalline structures.  In simulated annealing, a minimum (or maximum) value of some global "energy" function is sought.  This model attempts to find a maximal solution in a two-dimensional grid.

 We use such a simple problem in this model in order to highlight the solution technique only. 

## HOW IT WORKS

In this model, the objective function is defined for each patch in our 2D world. The location of each patch (x and y coordinates) can be thought of as the parameter values of the objective function. 

The optimization works as follows.  The system has a "temperature", which controls how much change is allowed to happen.  A random location for an initial solution is defined, and then for each step, a potential move to a new solution (location) is either accepted or rejected.  Changes that result in a greater solution value are always accepted (changes that result in no change of solution value will also always be accepted if the ACCEPT-EQUAL-CHANGES? switch is turned on). Changes that result in a lower solution value are only accepted with some probability, which is proportional to the "temperature" of the system using the Boltzmann Distribution.  The temperature of the system decreases over time, according to some cooling schedule, which means that initially changes that decrease solution values will often be accepted, but as time goes on they will be accepted less and less frequently.  This is similar to cooling a material slowly in natural annealing, to allow the molecules to settle into nice crystalline patterns.  Eventually the temperature approaches zero, at which point the simulated annealing method is equivalent to a random mutation hill-climber search, where only beneficial changes are accepted.

## HOW TO USE IT

Press the SETUP button to initialize the model and solution space.

Press the STEP button to go from one temperature to the next lower temperature.

Press the GO button to have the algorithm run until a solution has been found.

Adjust the COOLING-RATE slider to change how quickly the temperature drops.  
The current temperature is shown in the TEMPERATURE monitor.

The DELTA-MAX slider controls how far a potential movement can be.

If the ACCEPT-EQUAL-CHANGES? switch is ON, then the system will always accept a new solution that yields no change in solution value.  If it is OFF, then equal solutions are treated the same as those decrease the solution value, and thus are only accepted probabilistically based on the system temperature.

The Solution monitors and plot show how the algorithm is performing and the best solution that has been found.


## THINGS TO NOTICE

Slower cooling rates lead to higher optimal solutions (on average).

## THINGS TO TRY

If you turn ACCEPT-EQUAL-CHANGES? to ON, does slow cooling still work better than fast cooling?

Try varying the DELTA-MAX.  Does this help the system to reach more optimal configurations?

## EXTENDING THE MODEL

Currently, the probability of accepting a change that increases the best solution value is always 1, and the probability of accepting a change that decreases the solution value is based on the temperature of the system and the amount by which the solution has changed. Try extending the model to make more alternative acceptance decision criteria.

Simulated annealing can be used on a wide variety of optimization problems.  Experiment with using this technique on different "energy/cost" function, or even entirely different problems.

## NETLOGO FEATURES



## RELATED MODELS

Particle Swarm Optimization, Simple Genetic Algorithm, Crystallization Basic, Ising

## CREDITS AND REFERENCES

Original papers describing a simulated annealing  
S. Kirkpatrick and C. D. Gelatt and M. P. Vecchi, Optimization by Simulated Annealing, Science, Vol 220, Number 4598, pages 671-680, 1983.  
V. Cerny, A thermodynamical approach to the traveling salesman problem: an efficient simulation algorithm. Journal of Optimization Theory and Applications, 45:41-51, 1985


## HOW TO CITE

If you mention this model in a publication, we ask that you include these citations for the model itself and for the NetLogo software:

* Stonedahl, F. and Wilensky, U. (2009).  NetLogo Simulated Annealing model.  http://ccl.northwestern.edu/netlogo/models/SimulatedAnnealing.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
* Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT NOTICE

Copyright 2012 Kevin Brewer. All rights reserved.
Copyright 2009 Uri Wilensky. All rights reserved.

![CC BY-NC-SA 3.0](http://i.creativecommons.org/l/by-nc-sa/3.0/88x31.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Uri Wilensky at uri@northwestern.edu and Kevin Brewer at kbrewer@olivet.edu
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.2
@#$#@#$#@
need-to-manually-make-preview-for-this-model
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
