globals [
  resources        ;; natural resources available
  ritual-threshold ;; threshold to trigger rituals
  pig-count        ;; count of pigs
  human-count      ;; count of humans
  ritual-count     ;; count of rituals performed
  fertility-stage  ;; current fertility stage
  total-rituals    ;; total count of rituals performed
  rituals-per-tick ;; average rituals per tick
]

turtles-own [
  pig?             ;; indicates if the turtle is a pig
]

patches-own [
  resource-level   ;; resource level on each patch
]

to setup
  clear-all
  ;; Initial setup of agents and patches
  set-default-shape turtles "person"

  ;; Create the initial number of humans
  create-turtles init_humans [
    setxy random-xcor random-ycor
    set color black
    set pig? false
  ]
  ;; Create the initial number of pigs
  create-turtles init_pigs [
    setxy random-xcor random-ycor
    set color pink
    set shape "pig"
    set pig? true
  ]
  ;; Assign random resource levels to patches
  ask patches [
    set resource-level random 100
    patch-recolor
  ]
  ;; Calculate the total resources
  set resources sum [resource-level] of patches
  ;; Set the ritual threshold
  set ritual-threshold threshold_resources
  ;; Initial counts of pigs and humans
  set pig-count count turtles with [pig?]
  set human-count count turtles with [not pig?]
  ;; Reset the tick counter
  reset-ticks
  ;; Initialize the total ritual counter
  set total-rituals 0
  ;; Initialize population graphs
  initialize-population-plots
end

to patch-recolor ;; patch procedure
  ;; Color patches based on the amount of resources they have
  set pcolor (green + 4.9 - (resource-level / 25))
end

to go
  ;; Execute simulation steps
  if agricultural-cycles [
    switch-fertility-stages
  ]
  grow-resources
  grow-humans
  grow-pigs
  consume-resources
  check-ritual
  ;; Update pig and human counts
  set pig-count count turtles with [pig?]
  set human-count count turtles with [not pig?]
  ;; Recolor the patches
  ask patches [patch-recolor]
  ;; Calculate the average rituals per tick
  if ticks > 0 [
    set rituals-per-tick total-rituals / ticks
  ]
  ;; Advance one tick
  tick
  ;; Update population graphs
  update-population-plots
end

to grow-resources
  ;; Increase resources on all patches based on fertility stage and growth rate
  ask patches [
    if agricultural-cycles [
      if fertility-stage = "high" [
        set resource-level resource-level + (2 * resource-growth-rate)
      ]
      if fertility-stage = "low" [
        set resource-level resource-level + (1 * resource-growth-rate)
      ]
    ]
    if not agricultural-cycles [
      set resource-level resource-level + resource-growth-rate
    ]
    ;; Ensure resources do not go below zero
    if resource-level < 0 [
      set resource-level 0
    ]
  ]
  ;; Recalculate total resources
  set resources sum [resource-level] of patches
end

to switch-fertility-stages
  ;; Switch fertility stages every 50 ticks
  if ticks mod 50 = 0 [
    ask patches [
      ifelse fertility-stage = "high" [
        set fertility-stage "low"
      ] [
        set fertility-stage "high"
      ]
    ]
  ]
end

to grow-humans
  ;; Grow the human population if it's below 100 (arbitrary limit)
  if (count turtles with [not pig?]) < 100 [
    create-turtles 1 [
      setxy random-xcor random-ycor
      set color black
      set pig? false
    ]
  ]
end

to grow-pigs
  ;; Grow the pig population if it's below 200 (arbitrary limit)
  if (count turtles with [pig?]) < 200 [
    create-turtles 1 [
      setxy random-xcor random-ycor
      set color pink
      set shape "pig"
      set pig? true
    ]
  ]
end

to consume-resources
  ;; Consume resources by all turtles
  ask turtles [
    if any? patches with [resource-level > 0] [
      let target-patch one-of patches with [resource-level > 0]
      ask target-patch [
        set resource-level max (list 0 (resource-level - 10))
        set resources resources - 10
      ]
    ]
  ]
end

to check-ritual
  ;; Check if a ritual should be performed
  if resources < ritual-threshold [
    perform-ritual
  ]
end

to perform-ritual
  ;; Perform a ritual
  let pigs-to-kill count turtles with [pig?] / 2
  set ritual-count ritual-count + 1
  set total-rituals total-rituals + 1
  ;; Sacrifice pigs
  ask n-of pigs-to-kill turtles with [pig?] [
    die
  ]
  ;; Increase resources due to the ritual
  set resources resources + pigs-to-kill
  ;; Check if the human population should decrease
  if resources < ritual-threshold [
    perform-human-death
  ]
end

to perform-human-death
  ;; Decrease the human population (one-fifth of humans die if resources are below the threshold)
  let humans-to-kill count turtles with [not pig?] / 5
  ask n-of humans-to-kill turtles with [not pig?] [
    die
  ]
end

to initialize-population-plots
  ;; Initialize population graphs
  set-current-plot "Population"
  clear-plot
  set-current-plot-pen "Humans"
  set-plot-pen-color blue
  set-current-plot-pen "Pigs"
  set-plot-pen-color red
end

to update-population-plots
  ;; Update population graphs
  set-current-plot "Population"
  set-current-plot-pen "Humans"
  plot count turtles with [not pig?]
  set-current-plot-pen "Pigs"
  plot count turtles with [pig?]
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
647
448
-1
-1
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
54
73
117
106
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
54
144
117
177
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
664
20
864
170
Population
Time
NIL
0.0
10.0
0.0
200.0
true
true
"" ""
PENS
"humans" 1.0 0 -2674135 true "" ""
"pigs" 1.0 0 -13345367 true "" ""

MONITOR
662
188
719
233
Humans
human-count
17
1
11

MONITOR
663
237
720
282
Pigs
pig-count
17
1
11

SLIDER
24
214
196
247
init_pigs
init_pigs
0
100
20.0
1
1
NIL
HORIZONTAL

SLIDER
26
267
198
300
init_humans
init_humans
0
100
70.0
1
1
NIL
HORIZONTAL

MONITOR
664
290
789
335
Resources
resources
17
1
11

SLIDER
26
317
198
350
threshold_resources
threshold_resources
0
100000
7110.0
10
1
NIL
HORIZONTAL

PLOT
871
20
1071
170
Resources
Time
NIL
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"resources" 1.0 0 -10899396 true "" "plot resources"

MONITOR
722
188
865
233
Number of rituals executed
ritual-count
17
1
11

SWITCH
25
361
197
394
agricultural-cycles
agricultural-cycles
0
1
-1000

PLOT
872
188
1072
338
Rituals per ticks
Time
NIL
0.0
10.0
0.0
0.1
true
true
"" ""
PENS
"rituals per tick" 1.0 0 -2674135 true "" "plot rituals-per-tick"

MONITOR
728
236
834
281
Rituals per ticks
rituals-per-tick
5
1
11

SLIDER
26
408
198
441
resource-growth-rate
resource-growth-rate
0.0
5.0
0.2
0.2
1
NIL
HORIZONTAL

@#$#@#$#@
**Model Information**

**WHAT IS IT?**
This model simulates the dynamics between the human and pig populations in a community, inspired by anthropologist Roy Rappaport's studies on the Tsembaga-Maring. The model includes natural resource management, population growth, resource consumption, and rituals that affect both the pig and human populations. It also allows for alternating periods of high and low fertility, impacting resource regeneration.

**HOW IT WORKS**
The model operates as follows:

**Initialization (setup):**
- An initial number of humans and pigs are created according to user-set values via sliders.
- Each patch of the world is assigned a random resource level.
- The total available resources are calculated.
- Initial values for the pig, human, and ritual counters are set.
- Population graphs are initialized.
- The resource increment rate is set based on the slider value.

**Simulation (go):**
During each tick, the following actions are performed:
- **Resource Growth (grow-resources):** Resources in each patch increase based on the fertility stage and the growth rate.
- **Fertility Stage Switching (switch-fertility-stages):** If the agricultural cycles option is enabled, patches switch between high and low fertility every 50 ticks.
- **Human Population Growth (grow-humans):** If the human population is less than 100, a new human is created.
- **Pig Population Growth (grow-pigs):** If the pig population is less than 200, a new pig is created.
- **Resource Consumption (consume-resources):** Humans and pigs consume resources from the patches. Each consumption reduces the resource level by 10 units.
- **Ritual Check and Performance (check-ritual):** If available resources fall below the defined threshold (ritual-threshold), a ritual is performed where pigs are sacrificed, increasing the available resources. If resources are still insufficient after the ritual, the human population is reduced.
- **Rituals (perform-ritual):** Half of the pig population is sacrificed, and the available resources increase by an amount equivalent to the number of pigs sacrificed.
- **Human Population Reduction (perform-human-death):** If available resources are still insufficient after a ritual, the human population is reduced by one-fifth.

**HOW TO USE IT**

**Initial Setup:**
- Adjust the sliders for `init-humans`, `init-pigs`, `threshold-resources`, and `resource-growth-rate` to set the initial number of humans, pigs, the resource threshold for performing rituals, and the resource growth rate, respectively.
- Activate the `agricultural-cycles` switch to include agricultural cycles.

**Running the Model:**
- Click the `setup` button to initialize the model with the configured values.
- Click the `go` button to start the simulation. The `go` button should be set to run continuously.

**Monitoring and Graphs:**
- Observe the monitors to see the pig and human counters.
- Watch the graph to see the evolution of the pig and human populations over time.

**THINGS TO NOTICE**
- Observe how the human and pig populations interact with the available resources.
- Note how the performance of rituals affects population dynamics and resource availability.
- Observe the conditions under which the human population is reduced and how this impacts system stability.

**THINGS TO TRY**
- Adjust the initial values of humans and pigs to see how different configurations affect the model's dynamics.
- Modify the resource threshold (`ritual-threshold`) to observe how changes in ritual frequency impact population and resources.
- Experiment with turning agricultural cycles on and off to see how they affect resource availability and population dynamics.

**EXTENDING THE MODEL**
- Introduce new factors affecting population growth, such as diseases or migrations.
- Add variations in the resource growth rate to simulate different environmental conditions.
- Include new species or different types of resources to increase the model's complexity.

**CREDITS AND REFERENCES**
This model is based on ideas presented by Roy Rappaport in his book "Pigs for the Ancestors."

If you mention this model or the NetLogo software in a publication, please include the following citations:
- For the model itself: Based on Rappaport, R. (1968). *Pigs for the Ancestors: Ritual in the Ecology of a New Guinea People*. Yale University Press.
- Please cite the NetLogo software as: Wilensky, U. (1999). *NetLogo*. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

pig
false
13
Polygon -2064490 true true 196 203 206 250 188 250 182 208 171 189 142 189 92 188 71 185 68 221 45 219 44 191 33 159 34 135 42 114 56 100 100 84 171 80 193 81 221 85 236 85 266 100 281 131 251 131 238 128 220 177
Polygon -2064490 true true 258 91 290 91 245 106 233 89
Polygon -2064490 true true 70 214 83 255 64 259 45 212
Polygon -2064490 true true 44 110 80 103 47 124 26 92
Polygon -2064490 true true 37 86 27 96 48 86 45 83
Polygon -2064490 true true 45 83 57 95 47 83 56 92
Polygon -2064490 true true 49 89 42 80 63 90 60 93
Polygon -2064490 true true 35 84 26 95 46 84 43 81
Polygon -2064490 true true 35 84 26 95 46 84 43 81
Polygon -2064490 true true 35 84 26 95 46 84 43 81
Polygon -2064490 true true 30 87 29 97 37 106 38 84

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.4.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
