breed [ nodes node ]
nodes-own
[ 
  grammar
  grammar-bias
  spoken-val
]
globals [ 
  seed 
  initial-fraction-influenced-by-minority
  ]
to setup [ rseed ]
  ca
  set seed rseed
  random-seed seed
  with-local-randomness [  ask patches [set pcolor cyan - 3 ]  ]
  setup-nodes
  ifelse network-type = "two-communities"
  [
    ; set up community 1
    let nodesetA (nodes with [ who < (number-of-nodes / 2) ])
    setup-preferential-network nodesetA average-node-degree
    ; start all nodes with grammar 0
    ask nodesetA [ set grammar 0.0 ]
    ; If comA-start = "resistant", we start by having some node strongly
    ; biased towards grammar 0, and distribute negative bias outward from that node.
    ;  - This is as if there was already a 0 cascade that succeeded.
    ; If comA-start = "ripe", we start by having some node strongly biased
    ; toward grammar 1, and distribute positive bias outward from that node.
    ;  - In this case the community is "ripe" for a 1s cascade.
    setup-biases nodesetA (one-of nodesetA) (comA-start = "resistant")
    ask nodesetA [  set xcor (xcor / 2) - (world-width / 4)  ]    
    let nodesetB (nodes with [ who >= (number-of-nodes / 2) ])
    setup-preferential-network nodesetB average-node-degree
    ; start all nodes with grammar 1
    ask nodesetB [ set grammar 1.0 ]
    ; see note above about comA-start: comB start is similar.
    setup-biases nodesetB (one-of nodesetB) (comB-start != "resistant")
    ask nodesetB [ set xcor (xcor / 2) + (world-width / 4)  ]
    
    ; sort from low bias (against grammar 1) to high bias (in favor of grammar 1)
    let nodelistA sort-by [ [grammar-bias] of ?1 < [grammar-bias] of ?2 ] nodesetA
    let nodelistB sort-by [ [grammar-bias] of ?1 < [grammar-bias] of ?2 ] nodesetB
    if (comA-bridge-bias = 1)
    [ set nodelistA reverse nodelistA ]
    if (comB-bridge-bias = 1)
    [ set nodelistB reverse nodelistB ]
    
    repeat intercommunitylinks
    [
      ask (first nodelistA) [ create-link-with (first nodelistB) ]   
      set nodelistA but-first nodelistA
      set nodelistB but-first nodelistB
    ]
    
  ]
  [  
    if (network-type = "random")
    [ setup-random-network ]
    if (network-type = "spatial")
    [  setup-spatially-clustered-network ]
    if (network-type = "preferential")
    [  setup-preferential-network nodes average-node-degree ]
    
    ifelse (start-target = "influentials")
    [
      let sortednodes sort-by [[count link-neighbors] of ?1 > [count link-neighbors] of ?2 ] nodes
      repeat start-target-rank
       [ set sortednodes but-first sortednodes ]
      ask (turtle-set sublist sortednodes 0 num-start-with-G1) [ set grammar 1 ]
    ][
      ask n-of num-start-with-G1 nodes 
      [
        set grammar 1.0
      ]
    ]
    ; if there is more than one node from which the new grammar might spread, we pick one randomly
    let start-node max-one-of nodes [ grammar ]
    setup-biases nodes start-node false
  ]  
  with-local-randomness [
    set initial-fraction-influenced-by-minority sum [ count link-neighbors ] of nodes with [ grammar > 0.5 ] / (2 * count links )
    if visuals?
    [
      ask nodes 
      [ 
        color-by-grammar 
        size-by-degree
      ]
    ]
  ]
end
to-report uniform-symmetric-bias-list [ len ]
  let bias-list n-values floor (len / 2) [ -0.5 + random-float 1.0 ]
  set bias-list sentence bias-list (map [ 0 - ? ] bias-list )
  if (length bias-list != len)
    [ set bias-list fput 0 bias-list ]
  report bias-list
end
to-report random-normal-cutoff [ avg stdev xmin xmax ]
  let x random-normal avg stdev
  while [ x < xmin or x > xmax ] 
  [ set x random-normal avg stdev ]
  report x
end
to-report normal-symmetric-bias-list [ len ]
  let stdev 10 ^ bias-stdev-log10
  let bias-list n-values floor (len / 2) [ random-normal-cutoff 0 stdev -0.5 0.5 ]
  set bias-list sentence bias-list (map [ 0 - ? ] bias-list )
  if (length bias-list != len)
    [ set bias-list fput 0 bias-list ]
  report bias-list
end
to setup-nodes
  set-default-shape nodes "circle"
    
  create-nodes number-of-nodes
  [
    ; for visual reasons, we don't put any nodes *too* close to the edges
    setxy random-xcor * .95 random-ycor * .95
    set grammar 0.0
  ]
end
to setup-biases [ thenodes start-node reverse-order? ]
  let bias-list false ; this will cause an error if bias-dist wasn't a valid choice.
  if (bias-dist = "flat")
  [ set bias-list n-values (count thenodes) [ global-bias ]  ]
  if (bias-dist = "uniform-symmetric")
  [ set bias-list uniform-symmetric-bias-list (count thenodes) ]
  if (bias-dist = "normal-symmetric")
  [ set bias-list normal-symmetric-bias-list (count thenodes) ]
  let nodelist [self] of thenodes
  if (bias-target = "influentials")
  [
    set bias-list sort bias-list
    set nodelist sort-by [[count link-neighbors] of ?1 < [count link-neighbors] of ?2 ] thenodes
  ]
  if (bias-target = "nearby")
  [
    set bias-list sort bias-list
    set nodelist sort-by [[__network-distance start-node links] of ?1 > [__network-distance start-node links] of ?2 ] thenodes
  ]
  if (reverse-order?) 
    [ set bias-list reverse bias-list ]
  foreach nodelist
  [
    ask ?
    [
      set grammar-bias first bias-list
      set bias-list but-first bias-list
    ]
  ]
end
to setup-random-network
  ask nodes [ 
    ask nodes with [who > [who] of myself ]
    [
      if (random-float 1.0 < (average-node-degree / (number-of-nodes - 1) ))
      [ create-link-with myself ]
    ]
  ]
  if visuals?
  [
     repeat 40 [ do-network-layout nodes ]  
     rescale-network-to-world
  ]
end
to setup-spatially-clustered-network
  let num-links (average-node-degree * number-of-nodes) / 2
  while [count links < num-links ]
  [
    ask one-of nodes
    [
      let choice (min-one-of ((other turtles) with [ not link-neighbor? myself ]) [ distance myself ])
      if (choice != nobody)
        [ create-link-with choice ]
    ]
  ]
  ; make the network look a little prettier
  if visuals?
  [
     repeat 10 [ do-network-layout nodes ]  
     rescale-network-to-world
  ]
end

to setup-preferential-network [ thenodes avg-node-deg ]
  link-preferentially thenodes (avg-node-deg / 2)
  
  ; make the network look a little prettier
  if visuals?
  [
     with-local-randomness [
       layout-radial thenodes links (max-one-of thenodes [ count link-neighbors ] )
     ]
     repeat 10 [ do-network-layout thenodes ]  
     rescale-network-to-world
  ]
end
; The parameter k is the number of edges to add at each step (e.g. k=1 builds a tree)
;  (if k has a fractional part, then we probabilistically add either floork aor floork + 1 edges)
;  k MUST be 2 or greater, otherwise there are errors!
to link-preferentially [ nodeset k ]
  let floork (floor k)
  let fractionk (k - floork)
  let nodelist sort nodeset
  let neighborchoicelist sublist nodelist 0 floork
  
  ask item floork nodelist
  [ 
    create-links-with turtle-set neighborchoicelist 
    set neighborchoicelist sentence (n-values floork [ self ] ) neighborchoicelist
  ]
  
  foreach sublist nodelist (floork + 1) (length nodelist)
  [
    ask ?
    [
      let tempneighborlist neighborchoicelist
      let mydegree floork + ifelse-value ((who > floork + 1) and (random-float 1.0 < fractionk)) [ 1 ] [ 0 ]
      repeat mydegree
      [
        let neighbor one-of tempneighborlist
        set tempneighborlist remove neighbor tempneighborlist 
        set neighborchoicelist fput neighbor neighborchoicelist
        create-link-with neighbor
      ]
      set neighborchoicelist sentence (n-values mydegree [ self ] ) neighborchoicelist
    ]
  ]
end

to do-network-layout [ thenodes ]
   with-local-randomness [
     layout-spring thenodes links 0.3 0.8 * (world-width / (sqrt number-of-nodes)) 0.5
   ]
end
to rescale-network-to-world
    with-local-randomness [
      let minx (min [ xcor ] of nodes)
      let miny (min [ ycor ] of nodes)
      let cw (max [ xcor ] of nodes) - minx
      let ch (max [ ycor ] of nodes) - miny
      ask nodes [ 
        set xcor (xcor - minx) / cw * (world-width - 1) + min-pxcor
        set ycor (ycor - miny) / ch * (world-height - 1) + min-pycor
      ]
    ]
end
to go
  ask nodes [ speak ]
  ask nodes [ learn ]
;; this would be a different type of scheduling, where high degree nodes
;; are 'learning' much more quickly than the rest of the agents.
;; if we delete this stuff, also delete "learn-from" procedure down below!
;  ask links [
;    ask both-ends [
;      speak
;      learn-from other-end
;    ]
;  ]
  
  if visuals?
  [
    with-local-randomness [
      ask nodes [ color-by-grammar ]
    ]
    update-plot
  ]
  tick
end
to size-by-degree
  set size 0.3 * sqrt (count link-neighbors + 1)
end
to color-by-grammar
  set color scale-color yellow grammar 0 1.000000001
end
to color-by-grammar-bias
  set color scale-color red grammar-bias -0.50000001 .500000001
end
to-report sigmoid-func [ x nonlinearity repel-offset ]
  ; this is a sigmoid-type function [0,1] --> [0,1] with parameters:
  ;    x: input
  ;    nonlinearity: degree of nonlinearity (0 = linear, 100 = step function)
  ;    repel-offset: determines (repelling) fixed point: x' = 0.5 + repel-offset
  if nonlinearity = 100 [report ifelse-value (x < (0.5 + repel-offset)) [0.0] [1.0]]
  if nonlinearity = 0 [report x] ; linear!
  if (repel-offset < -0.5) [ set repel-offset -0.5 ]
  if (repel-offset > 0.5) [ set repel-offset 0.5 ]
  let a (nonlinearity / (100.0 - nonlinearity))
  let left-term (x * exp(a * (x - repel-offset)))
  let right-term ((1.0 - x) * exp(a * (1.0 - x + repel-offset)))
  report (left-term / (left-term + right-term))
end

to speak
  let prob (sigmoid-func grammar categoricalness (0.0 - (global-bias + grammar-bias)))
  ifelse (probabilistic-speech?)
  [   set spoken-val ifelse-value (random-float 1.0 < prob) [ 1 ] [ 0 ]   ]
  [   set spoken-val prob   ]
end
to learn
  if (not any? link-neighbors)
    [ stop ]
  let new-gram (learning-rate * mean [ spoken-val ] of link-neighbors) + (1 - learning-rate) * grammar 
  ifelse (new-gram > 1) 
    [ set new-gram 1 ]
    [ if (new-gram < 0) [ set new-gram 0 ] ]
  set grammar new-gram
end
;; This procedure would be useful, if we decided to use the different update scheduling mentioned in
;; the GO procedure, wherein high degree nodes do a lot more speaking *AND* learning than other nodes.
;to learn-from [ othernode ]
;  let new-gram (learning-rate * [ spoken-val ] of othernode) + (1 - learning-rate) * grammar 
;  ifelse (new-gram > 1) 
;    [ set new-gram 1 ]
;    [ if (new-gram < 0) [ set new-gram 0 ] ]
;  set grammar new-gram
;end

to update-plot
  with-local-randomness [
    set-current-plot "Grammar State"
    set-current-plot-pen "state"
    plot mean [ grammar ] of nodes
    set-current-plot-pen "spoken"
    plot mean [ spoken-val ] of nodes
  ]
end
to-report converged?
  ; if the chance of the out-lier node producing a minority-grammar
  ;    token in the next 10,000 time steps is safely less than 0.01%, then stop.
    if not any? nodes [ report false ]
    report ((min [ grammar ] of nodes) > (1 - 1E-8) or (max [ grammar ] of nodes) < 1E-8)
end
;; The following several procedures are not necessary for the running of the model, but may be
;; useful for measuring the model, BehaviorSpace experiments, etc.
to-report cascaded?
  ifelse (converged? and mean [grammar] of nodes > 0.5) 
    [ report 1 ] 
    [ report 0 ]
end
to-report cascaded90?
  ifelse (mean [grammar] of nodes > 0.9)
  [ report 1 ] 
  [ report 0 ]
end
to-report communityA
  report nodes with [ who < (count nodes / 2) ]
end
to-report communityB
  report nodes with [ who >= (count nodes / 2) ]
end
;; The following procedures are not at all crucial to the model
;; I just used them to be able to repeat some interesting setups,
;; do some quick testing, etc.  
;; They should probably all be deleted at some point. ~Forrest 7/22/2008
to demo-setup
  clear-all
set number-of-nodes 100
set average-node-degree 3
set bias-dist "normal-symmetric"
set bias-stdev-log10 1
set probabilistic-speech? true
set network-type "preferential"
set categoricalness 50
set global-bias 0
set num-start-with-G1 25
set learning-rate 0.05
set visuals? true
setup 367808704
end
to demo-setup2
; if we use robert's type of node interactions in each time step:
; seed that cascades, but only for high learning rate
setup 762758417 
set learning-rate 0.6 
end
;to setup-one [theseed]
;clear-all
;setup theseed
;ask nodes [ set grammar 0 ]
;;ask max-n-of (new-grammar-fraction * number-of-nodes) nodes  [ count link-neighbors ] [ set grammar 1 ]
;ask max-n-of 1 nodes  [ count link-neighbors ] [ set grammar 1 ]
;;print mean [ count link-neighbors ] of nodes with [ grammar > 0.5 ]
;ask nodes [ color-by-grammar ]
;end
to demo-setup-one
set learning-rate 0.13
set visuals? true
set average-node-degree 3
set probabilistic-speech? true
set categoricalness 50
set bias-target "influentials"
set bias-dist "uniform-symmetric"
set number-of-nodes 200
set network-type "preferential"
set bias-stdev-log10 1
set global-bias 0
set num-start-with-G1 2
set start-target "influentials"
set start-target-rank 0
  setup 1324523410
;  for N=300, out of 1000 runs, these 3 seeds cascaded upwards:  
;  SEED: 6410918, 1256638123,  685548220
end
to-report quicktest
  setup new-seed 
  repeat 1000 [ 
    go 
    if converged? [ report mean [grammar] of nodes]
  ]
  report mean [ grammar ] of nodes
end
to demo-surprising-case
  set bias-stdev-log10 1
  set number-of-nodes 300
  set start-target "none"
  set network-type "preferential"
  set learning-rate 0.05
  set bias-target "nearby"
  set categoricalness 50
  set global-bias 0
  set bias-dist "normal-symmetric"
  set visuals? true
  set average-node-degree 4
  set num-start-with-G1 3
  set probabilistic-speech? true
  set start-target-rank 0
  setup 3543911878112519 
end
@#$#@#$#@
GRAPHICS-WINDOW
236
10
653
448
18
18
11.0
1
10
1
1
1
0
0
0
1
-18
18
-18
18
1
1
1
ticks
CC-WINDOW
5
604
956
699
Command Center
0
BUTTON
10
290
80
325
setup
clear-all\\nsetup new-seed
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
BUTTON
85
290
150
325
go
go\\nif (converged?) [ stop ]
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
PLOT
695
55
947
245
Grammar State
time
grammar
0.0
52.0
0.0
1.0
true
true
PENS
"spoken" 1.0 0 -8630108 true
"state" 1.0 0 -16777216 true
SLIDER
11
72
206
105
number-of-nodes
number-of-nodes
10
500
100
1
1
NIL
HORIZONTAL
SLIDER
12
346
207
379
categoricalness
categoricalness
0
100
50
0.5
1
NIL
HORIZONTAL
SLIDER
10
110
205
143
average-node-degree
average-node-degree
2
10
3
1
1
NIL
HORIZONTAL
SLIDER
10
205
205
238
num-start-with-G1
num-start-with-G1
0
number-of-nodes
1
1
1
NIL
HORIZONTAL
SLIDER
11
382
206
415
learning-rate
learning-rate
0
1
0.1
0.001
1
NIL
HORIZONTAL
MONITOR
865
10
940
55
max-grammar
max [ grammar ] of nodes
4
1
11
MONITOR
780
10
865
55
mean-grammar
mean [ grammar ] of nodes
4
1
11
BUTTON
155
290
218
325
step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
MONITOR
705
10
780
55
min-grammar
min [grammar] of nodes
4
1
11
SLIDER
675
266
830
299
global-bias
global-bias
-0.5
0.5
0
0.001
1
NIL
HORIZONTAL
CHOOSER
40
10
181
55
network-type
network-type
"spatial" "random" "preferential" "two-communities"
2
SLIDER
675
411
830
444
bias-stdev-log10
bias-stdev-log10
-5
1
1
0.25
1
NIL
HORIZONTAL
SWITCH
25
420
192
453
probabilistic-speech?
probabilistic-speech?
0
1
-1000
BUTTON
520
450
610
483
layout
with-local-randomness \\n[ do-network-layout nodes display ] 
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
MONITOR
855
335
941
380
mean-bias
mean [grammar-bias] of nodes
10
1
11
BUTTON
370
450
495
483
color-by-bias
with-local-randomness \\n[ ask nodes [ color-by-grammar-bias ] ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
SWITCH
255
450
358
483
visuals?
visuals?
0
1
-1000
CHOOSER
675
356
809
401
bias-dist
bias-dist
"flat" "uniform-symmetric" "normal-symmetric"
1
MONITOR
855
285
940
330
min-bias
min [grammar-bias] of nodes
4
1
11
CHOOSER
676
304
814
349
bias-target
bias-target
"influentials" "nearby" "none"
1
CHOOSER
40
155
178
200
start-target
start-target
"influentials" "none"
0
SLIDER
10
240
205
273
start-target-rank
start-target-rank
0
number-of-nodes - num-start-with-G1
0
1
1
NIL
HORIZONTAL
BUTTON
370
485
495
518
color-by-grammar
ask nodes [ color-by-grammar ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
SLIDER
20
460
192
493
intercommunitylinks
intercommunitylinks
0
number-of-nodes / 2
4
1
1
NIL
HORIZONTAL
CHOOSER
10
495
115
540
comA-bridge-bias
comA-bridge-bias
0 1
0
CHOOSER
125
495
230
540
comB-bridge-bias
comB-bridge-bias
0 1
0
MONITOR
675
455
811
500
avg deg
2 * count links / number-of-nodes
17
1
11
CHOOSER
10
545
115
590
comA-start
comA-start
"ripe" "resistant"
1
CHOOSER
125
545
230
590
comB-start
comB-start
"ripe" "resistant"
1
@#$#@#$#@
VERSION
-------
$Id: DiffusionOfLanguage.nlogo 39017 2008-03-29 22:31:47Z fjs750 $

WHAT IS IT?
-----------
This is a linguistics model about how a language change may (or may not) diffuse through a social network.  The key research question that it is interested in investigating is this:  How can initially rare grammar variants become dominant in a population, without any global bias in their favor?  It is known that such changes can and do occur in the real world - but what conditions are necessary to produce this behavior?  This model demonstrates that the behavior can be reproduced through the use of simple cognitively-motivated agent rules combined with a social network structure and certain distributions of heterogeneous bias in the population.  The language cascade occurs despite the fact that all of the agents' biases sum to 0.
While the model was developed for linguistics, there are potentially useful lessons to be learned about the interactions of heterogeneous agents in a social network, which may perhaps be applied to other disciplines, such epidemiology, or the diffusion of innovation in marketing.
HOW IT WORKS
------------
In this model, there are two opposing grammar variants (G0 and G1) in the population.  Each agent's grammar value lies on the range between 0.0 and 1.0.  The value 0.0 means that the agent only speaks grammar variant G0, whereas 1.0 means that the agent only speaks grammar variant G1.  For grammar values between 0.0 and 1.0, an agent may speak either G0 or G1, with some probability.  The probability is determined by a "production function", the shape of which depends on the CATEGORICALNESS parameter, as well as a 'bias' which can vary between agents (this 'bias' may be distributed in various ways, as shall be discussed in more detail later).  It is called a "production" function because it maps a tendency toward one grammar or another into a probability of producing a token for one grammar or the other.  If CATEGORICALNESS = 0, the production function is linear, meaning that agents produce G1 tokens with probability given directly by their grammar value, and G0 tokens otherwise.  If CATEGORICALNESS > 0
the production function is nonlinear (in particular, sigmoidal).   The agent's bias determines the point at which the production function crosses the line Y = X, which may be considered repelling point, because if the agent's grammar value is below this repelling point and the agent were talking only to itself, it would eventually end up with grammar value 0.0, but if the grammar value started above this point, it would eventually end up at grammar value 1.0.  The larger the CATEGORICALNESS parameter, the closer the sigmoidal production function is to a step function, and at CATEGORICALNESS = 100, the production function actually becomes a step function.  This means that if the agents grammar value is above a point (determined by its bias) it will only speak G1, and if it is below that point, it will only speak G0.  In this case, agents are completely categorical about their speech, and are unwilling to mix the usage of two the two competing grammars.
Over time each agent updates the state of its internal grammar value based on the tokens it is hearing from neighboring agents in the social network.  More specifically, in each tick, each agent first produces a single token probabilistically, based on their grammar state and their production function.  Each agent then updates their grammar state to be closer to the mean grammar value that they heard from all of their neighbors.  We use what is sometimes called "alpha learning", whereby the new grammar state is a weighted average  of the old grammar state with the mean of all the tokens produced by the neighbors.  Thus, high degree nodes (agents) in the network (which we refer to as "influentials") are considered to be "heard" by many more nodes than low-degree nodes.  However, the LEARNING-RATE (rate of change from the current grammar state toward the perceived grammar of the neighbors) of all of the nodes is the same.
As an example, an agent that start with grammar value 1.0 will certainly produce a G1 grammar token in the first tick of the model.  After one tick, it may have heard G0 tokens from their neighbors, and have adjusted their grammar value downward, meaning that the probability of producing G1 is no longer 100%.  However, if the LEARNING-RATE is not too large, the agent's grammar value will probably still be quite high, which corresponds to a high likelihood of producing a G1 token in the next tick.  However, over time the grammar value may undergo significant changes.

HOW TO USE IT / MODEL PARAMETERS
-------------
While the basic mechanics of the model are described simply above, there are numerous parameters, and ways to initialize or setup the model, to address different questions.
Here is a brief explanation of each parameter of control for the model, and how they related to the initialization and running of the model.
The social network structure (NETWORK-TYPE) may be initialized in several ways:
 * "spatial" causes nearby agents (in Euclidean space) to be linked together
 * "random" refers to Erdos-Renyi random graphs
 * "preferential" refers to the Barabasi-Albert preferential attachment method of creating scale-free networks.  The method has been extended slightly to handle the creation of networks with odd average degree, by probabilistically choosing to add either K or K+1 edges as each new node is attached to the network.
 * "two-communities" consists of two "preferential" networks loosely connected to each other by some number of links (specified by the INTERCOMMUNITYLINKS parameter).
The network is created with the specified NUMBER-OF-NODES and AVERAGE-NODE-DEGREE.
By default, nodes start with an internal grammar value of 0.0, meaning they have no chance of ever using variant G1.  The NUM-START-WITH-G1 parameter, however, controls the number of nodes in the network that start with grammar value 1.0.  
If START-TARGET = "none", the agents are randomly chosen to start with grammar value 1.0.  But if START-TAGET = "influentials", then the 1.0 grammar value is assigned by starting with the START-TARGET-RANK most influential agent and going down in order.  For instance, if START-TARGET-RANK = 9, and NUM-START-WITH-G1 = 3, then the 10th, 11th, and 12th most influential agents (highest-degree nodes) will be assigned grammar value 1.0.
Each agent is assigned a bias toward one grammar variant or the other. The bias can  range from +0.5 (strongly supporting G1) to -0.5 (strongly supporting G0).  If BIAS-DIST = "flat", then all agents are assigned the same bias.  If BIAS-DIST = "uniform-symmetric", then the biases are chosen symmetrically in pairs (X and -X) from a uniform distribution between -0.5 and 0.5.  If BIAS-DIST = "normal-symmetric", then the biases are chosen symmetrically in pairs (X and -X) from a normal distribution, centered around 0, and with the log (base 10) of the standard deviation given by BIAS-STDEV-LOG10 parameter.  The distribution is truncated at -0.5 and 0.5 (if a value is out of range, we redraw from the distribution).
Additionally, all agents' biases are affected by the GLOBAL-BIAS parameter.
The BIAS-TARGET parameter controls how bias is distributed in the social network.  If BIAS-TARGET = "none", then bias is randomly distributed.  If BIAS-TARGET = "nearby", then bias is distributed in sorted order (positive bias down to negative) starting with the most influential agent, down to the least influential agent.  If BIAS-TARGET = "nearby", then bias is distributed in sorted order outward from a random one of the agents that is starting with the G1 grammar.  This last method has the effect of creating a very favorable initial audience for this G1 speakers, and (from our experiments) appears to greatly improve the chances of a language cascade.
The preceding discussion is most relevant for the "spatial", "random", and "preferential" network types.  The grammar states and biases for the "two-communities" network-type are initialized according to different rules.  In this case, two "preferential" network communities are created - one consisting initially of all G0 speakers and the other consisting of all G1 speakers.  The COMA-START and COMB-START parameters control whether the bias is distributed in such a way that the community is more ripe for a language cascade to occur, or more resistant against change to the status quo.  More specifically, in each community, the biases are distributed outward from a random node in sorted order (either up, or down, depending). In Community A, if the bias is distributed outward starting with positive bias (supporting G1) down to negative bias, then the network will be more "ripe" for a G1 cascade.  On the other hand, distributing bias from negative bias (supporting G0) outward to positive bias will create a configuration that is more resistant to change.  For Community B (which starts with G1 prevalent) the situation is reversed, but otherwise exactly the same.  
The links between these two communities are chosen based on the COMA-BRIDGE-BIAS and COMB-BRIDGE-BIAS parameters.  If COMA-BRIDGE-BIAS = 0, then the agents in Community A that are most biased towards G0 will be chosen as "bridge" nodes - meaning they will be linked to the other community.  If COMA-BRIDGE-BIAS = 1, then the agents most biased towards G1 will be bridge nodes.  Similarly, COMB-BRIDGE-BIAS determines which nodes will be bridge nodes in Community B.
As mentioned above, the CATEGORICALNESS parameter affects the degree to which nodes are willing to speak the two grammar variants interchangeably, rather than having a stronger preference to speak consistently, or semi-categorically.
The LEARNING-RATE parameter controls the rate at which agents speaking affects other agents internal grammar values.  The grammar value of each agent is updated by a weighted sum of the old grammar value and the mean heard grammar of its neighbors. The LEARNING-RATE is the weight given to new data, while (1 - LEARNING-RATE) is the weight given to the old grammar value.
The PROBABALISTIC-SPEECH? parameter controls whether agents always speak 0 or 1 tokens probabilistically (ON), or else speak the real-valued numbers between 0 and 1 produced by their production functions (OFF).  The default is for PROBABALISTIC-SPEECH? to be ON.  However, turning it OFF could correspond to a longer iterated batch learning process.  In many ways, turning it OFF has the effect of removing some noise from the system, and causing faster convergence to an equilibrium.  However, the noise *can* be crucial in certain situations, and the behavior will be different.  There may be some interesting avenues for further research here...
The VISUALS? parameter turns on or off the visual display.  Turning VISUALS? OFF can help speed up the runs when running experiments.  It will not effect the outcome of the model at all.
The COLOR-BY-BIAS and COLOR-BY-GRAMMAR buttons affect the visualization of the network, scaling low values (0.0 grammar, or -0.5 bias) to black, and high values (1.0 grammar, 0.5 bias) to white.
The LAYOUT button can be used to try to improve the visual aesthetics of the network layout.  Note that this only affects visualization, and does not affect the model itself.
The SETUP button initializes the model, and the GO button runs the simulation until it has converged (or forever, if it does not converge).  The STEP button goes one tick at a time.
Various monitors show statistics (min, max, mean) for the grammar values or grammar biases.  The "GRAMMAR STATE" plot also plots the mean internal grammar value of the agents over time, as well as the mean spoken value.
THINGS TO NOTICE
--------------------

THINGS TO TRY
-------------

EXTENDING THE MODEL
-------------------
RELATED MODELS
----------------------
Language Change (by Celina Troutman)
CREDITS AND REFERENCES
----------------------
Written by Forrest Stonedahl, in collaboration with Janet Pierrehumbert and Robert Daland.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250
airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15
arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150
box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75
bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30
butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60
car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58
circle
false
0
Circle -7500403 true true 0 0 300
circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123
cylinder
false
0
Circle -7500403 true true 0 0 300
dot
false
0
Circle -7500403 true true 90 90 120
face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240
face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225
face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183
fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30
flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45
flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240
house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120
leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195
line
true
0
Line -7500403 true 150 0 150 300
line half
true
0
Line -7500403 true 150 0 150 150
pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120
person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105
plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90
square
false
0
Rectangle -7500403 true true 30 30 270 270
square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240
star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108
target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60
tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152
triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255
triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224
truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42
turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99
wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269
x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 4.0.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="explore_n50" repetitions="100" runMetricsEveryStep="false">
    <setup>setup random (2 ^ 31)</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <exitCondition>converged?</exitCondition>
    <metric>converged?</metric>
    <metric>round (mean [grammar] of nodes)</metric>
    <metric>mean [grammar] of nodes</metric>
    <metric>min [grammar] of nodes</metric>
    <metric>max [grammar] of nodes</metric>
    <metric>count nodes with [ grammar &gt; 0.5 ]</metric>
    <metric>initial-fraction-influenced-by-minority</metric>
    <metric>seed</metric>
    <metric>[grammar] of nodes</metric>
    <steppedValueSet variable="learning-rate" first="0.02" step="0.04" last="0.18"/>
    <enumeratedValueSet variable="bias-dist">
      <value value="&quot;uniform-symmetric&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="visuals?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-grammar-fraction">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probabilistic-speech?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-stdev-log10">
      <value value="1"/>
    </enumeratedValueSet>
    <steppedValueSet variable="average-node-degree" first="2" step="2" last="10"/>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="categoricalness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-bias">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="explore_n150" repetitions="100" runMetricsEveryStep="false">
    <setup>setup random (2 ^ 31)</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <exitCondition>converged?</exitCondition>
    <metric>converged?</metric>
    <metric>round (mean [grammar] of nodes)</metric>
    <metric>mean [grammar] of nodes</metric>
    <metric>min [grammar] of nodes</metric>
    <metric>max [grammar] of nodes</metric>
    <metric>count nodes with [ grammar &gt; 0.5 ]</metric>
    <metric>initial-fraction-influenced-by-minority</metric>
    <metric>seed</metric>
    <metric>[grammar] of nodes</metric>
    <steppedValueSet variable="learning-rate" first="0.02" step="0.04" last="0.18"/>
    <enumeratedValueSet variable="bias-dist">
      <value value="&quot;uniform-symmetric&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="visuals?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-grammar-fraction">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probabilistic-speech?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-stdev-log10">
      <value value="1"/>
    </enumeratedValueSet>
    <steppedValueSet variable="average-node-degree" first="2" step="2" last="10"/>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="150"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="categoricalness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-bias">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="explore_n500" repetitions="100" runMetricsEveryStep="false">
    <setup>setup random (2 ^ 31)</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <exitCondition>converged?</exitCondition>
    <metric>converged?</metric>
    <metric>round (mean [grammar] of nodes)</metric>
    <metric>mean [grammar] of nodes</metric>
    <metric>min [grammar] of nodes</metric>
    <metric>max [grammar] of nodes</metric>
    <metric>count nodes with [ grammar &gt; 0.5 ]</metric>
    <metric>initial-fraction-influenced-by-minority</metric>
    <metric>seed</metric>
    <metric>[grammar] of nodes</metric>
    <steppedValueSet variable="learning-rate" first="0.02" step="0.04" last="0.18"/>
    <enumeratedValueSet variable="bias-dist">
      <value value="&quot;uniform-symmetric&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="visuals?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-grammar-fraction">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probabilistic-speech?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-stdev-log10">
      <value value="1"/>
    </enumeratedValueSet>
    <steppedValueSet variable="average-node-degree" first="2" step="2" last="10"/>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="categoricalness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-bias">
      <value value="0"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="unittest1" repetitions="1" runMetricsEveryStep="false">
    <setup>setup random (2 ^ 31)</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <exitCondition>converged?</exitCondition>
    <metric>seed</metric>
    <metric>converged? and (min [grammar] of nodes) &gt; 0.5</metric>
    <enumeratedValueSet variable="learning-rate">
      <value value="0.02"/>
      <value value="0.1"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-dist">
      <value value="&quot;uniform-symmetric&quot;"/>
      <value value="&quot;normal-symmetric&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="visuals?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-grammar-fraction">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probabilistic-speech?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-stdev-log10">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-node-degree">
      <value value="2"/>
      <value value="3"/>
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="30"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="categoricalness">
      <value value="50"/>
      <value value="80"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-bias">
      <value value="0.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="unittest2" repetitions="1" runMetricsEveryStep="false">
    <setup>setup random (2 ^ 31)</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <exitCondition>converged?</exitCondition>
    <metric>seed</metric>
    <metric>converged? and (max [grammar] of nodes) &lt; 0.5</metric>
    <enumeratedValueSet variable="learning-rate">
      <value value="0.02"/>
      <value value="0.1"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-dist">
      <value value="&quot;uniform-symmetric&quot;"/>
      <value value="&quot;normal-symmetric&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="visuals?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-grammar-fraction">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probabilistic-speech?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-stdev-log10">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-node-degree">
      <value value="2"/>
      <value value="3"/>
      <value value="6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="30"/>
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="categoricalness">
      <value value="50"/>
      <value value="80"/>
      <value value="100"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-bias">
      <value value="-0.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="influential1" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup-one random (2 ^ 31)</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <exitCondition>converged?</exitCondition>
    <metric>cascaded?</metric>
    <metric>converged?</metric>
    <metric>round (mean [grammar] of nodes)</metric>
    <metric>mean [grammar] of nodes</metric>
    <metric>min [grammar] of nodes</metric>
    <metric>max [grammar] of nodes</metric>
    <metric>count nodes with [ grammar &gt; 0.5 ]</metric>
    <metric>initial-fraction-influenced-by-minority</metric>
    <metric>seed</metric>
    <enumeratedValueSet variable="probabilistic-speech?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-node-degree">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="learning-rate">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="visuals?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="categoricalness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-dist">
      <value value="&quot;uniform-symmetric&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-bias">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="new-grammar-fraction">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-influentials?">
      <value value="true"/>
    </enumeratedValueSet>
    <steppedValueSet variable="number-of-nodes" first="50" step="50" last="500"/>
    <enumeratedValueSet variable="bias-stdev-log10">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="targetinf" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup new-seed</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <exitCondition>converged?</exitCondition>
    <metric>cascaded?</metric>
    <metric>cascaded90?</metric>
    <metric>converged?</metric>
    <metric>round (mean [grammar] of nodes)</metric>
    <metric>mean [grammar] of nodes</metric>
    <metric>min [grammar] of nodes</metric>
    <metric>max [grammar] of nodes</metric>
    <metric>count nodes with [ grammar &gt; 0.5 ]</metric>
    <metric>initial-fraction-influenced-by-minority</metric>
    <metric>seed</metric>
    <enumeratedValueSet variable="bias-target">
      <value value="&quot;influentials&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="100"/>
      <value value="200"/>
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-node-degree">
      <value value="2"/>
      <value value="3"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-target-rank">
      <value value="1"/>
      <value value="4"/>
      <value value="16"/>
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-target">
      <value value="&quot;influentials&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-start-with-G1">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="learning-rate">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="categoricalness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-bias">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-dist">
      <value value="&quot;uniform-symmetric&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="visuals?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probabilistic-speech?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-stdev-log10">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="targetnearby" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup new-seed</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <exitCondition>converged?</exitCondition>
    <metric>cascaded?</metric>
    <metric>cascaded90?</metric>
    <metric>converged?</metric>
    <metric>round (mean [grammar] of nodes)</metric>
    <metric>mean [grammar] of nodes</metric>
    <metric>min [grammar] of nodes</metric>
    <metric>max [grammar] of nodes</metric>
    <metric>count nodes with [ grammar &gt; 0.5 ]</metric>
    <metric>initial-fraction-influenced-by-minority</metric>
    <metric>seed</metric>
    <enumeratedValueSet variable="bias-target">
      <value value="&quot;nearby&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="100"/>
      <value value="200"/>
      <value value="400"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-node-degree">
      <value value="2"/>
      <value value="3"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-target-rank">
      <value value="1"/>
      <value value="4"/>
      <value value="16"/>
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-target">
      <value value="&quot;influentials&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-start-with-G1">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;preferential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="learning-rate">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="categoricalness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-bias">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-dist">
      <value value="&quot;uniform-symmetric&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="visuals?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probabilistic-speech?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-stdev-log10">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="twocommunities" repetitions="100" runMetricsEveryStep="false">
    <setup>setup new-seed</setup>
    <go>go</go>
    <timeLimit steps="10000"/>
    <exitCondition>converged?</exitCondition>
    <metric>seed</metric>
    <metric>converged?</metric>
    <metric>mean [grammar] of nodes</metric>
    <metric>mean [grammar] of communityA</metric>
    <metric>mean [grammar] of communityB</metric>
    <metric>median [grammar] of nodes</metric>
    <metric>median [grammar] of communityA</metric>
    <metric>median [grammar] of communityB</metric>
    <metric>min [grammar] of nodes</metric>
    <metric>min [grammar] of communityA</metric>
    <metric>min [grammar] of communityB</metric>
    <metric>max [grammar] of nodes</metric>
    <metric>max [grammar] of communityA</metric>
    <metric>max [grammar] of communityB</metric>
    <metric>count nodes with [ grammar &gt; 0.5 ]</metric>
    <metric>count communityA with [ grammar &gt; 0.5 ]</metric>
    <metric>count communityB with [ grammar &gt; 0.5 ]</metric>
    <enumeratedValueSet variable="intercommunitylinks">
      <value value="1"/>
      <value value="2"/>
      <value value="4"/>
      <value value="8"/>
      <value value="16"/>
      <value value="32"/>
      <value value="64"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comA-bridge-bias">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comB-bridge-bias">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comB-start">
      <value value="&quot;resistant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="comA-start">
      <value value="&quot;resistant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="network-type">
      <value value="&quot;two-communities&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="number-of-nodes">
      <value value="256"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="learning-rate">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-target">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-start-with-G1">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="probabilistic-speech?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-dist">
      <value value="&quot;uniform-symmetric&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-stdev-log10">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="categoricalness">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="start-target-rank">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="global-bias">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="average-node-degree">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="visuals?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="bias-target">
      <value value="&quot;nearby&quot;"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
