globals
[ 
  errors!
  iterations
  ;; iterations-limit ;; (slider)
]

breeds
[
  nodes ;; entities in the system
  edges ;; the relationships between entities

]

turtles-own
[

]

nodes-own
[ name
  old-value
  value      
  trend ;; +1 or -1 or 0
  has-lower-limit?
  lower-limit
  has-upper-limit?
  upper-limit
  inlets  ;; the set of edges connecting to nodes influencing this node 
  outlets ;; the set of edges connecting to nodes influenced by this node
]

edges-own
[ name ;; allegedly used for labelling, e.g. "more fishing decreases fish" does not really appear in output
  inlet ;; node who's trend is being tracked
  outlet ;; node affected by the trend
  direction ;; +1 or -1, positive or negative relationship
  magnitude ;; degree to which trend affects outlet, relative to all the inlets of that outlet
  influence ;; magnitude * direction... pre-calculate in setup!
  net-influence ;; magnitude * direction * trend (from node)
]

;; In setup, the nodes and edges are defined
;; It might be fun to also have a drag/drop interface for this...
;; but that is candy.

to setup
   clear-all
   ask patches [ set pcolor gray + 4 ]
   set iterations 0
   
   ;;
   ;; STEP ONE
   ;; Ceate local variable for each node.
   ;; The variable contains the name of the thing the node represents.
   ;; By using local variables here, we allow NetLogo to spell-check our references to the
   ;; nodes later. Whereas, if we just used the strings everywhere, misspellings can cause
   ;; frustration. Also allows one to use SHORT symbols in the code, but have LONG
   ;; descriptive names in the display and output.
   ;;
   
   let suscept "Susceptibles"
   let immune "Immune"
   let ill "Diseased"
   let dead "Dead"
   ;; let poachers "poachers"
   
   ;;
   ;;STEP 2
   ;;Define the nodes.
   ;; The create-node helper procedure does this for us.
   ;; The parameters are: 
   ;; Name: A string. This is used both as a label, and as a tag used to locate and associate the nodes
   ;; Value: A number. This is the initial value of the node
   ;; Trend: A number. This is the initial trend of the node (that is, is the node currently increasing or decreasing?)
   ;; Lower-Limit: A list. This is used to define a lower limit for the nodes value, if any.
   ;;      The list may have two forms:
   ;;        If there IS a lower limit, then the list must be : [ true LIMIT ] where true is the boolean constant true, and LIMIT is a number.
   ;;        If there IS NOT a lower limit, then the list must be: [ false ]
   ;;        To make this easier and the meaning clearer in code, use the helper reporters provided.
   ;; Upper-Limit: A list. This is used to define an upper limit for the node's value, if any.
   ;;       The list is the same format as the Lower-imit.
   ;; 
   ;; The Limit helpers are:
   ;;      (no-lower-limit)
   ;;      (lower-limit-is NUMBER)
   ;;      (no-upper-limit)
   ;;      (upper-limit-is NUMBER)

   create-node: suscept      400   1  (lower-limit-is 0) (no-upper-limit) 
   create-node: immune        25   1  (lower-limit-is 0) (no-upper-limit)
   create-node: ill           25   1  (lower-limit-is 0) (no-upper-limit)
   create-node: dead           0   1  (lower-limit-is 0) (no-upper-limit)
  ;; create-node: poachers     100   1  (lower-limit-is 0) (no-upper-limit)

   ;;
   ;; Leave this line alone. If any node-name mismatches occur, this variable is changed.
   set errors! false
   
   ;; STEP 3
   ;; Define the rules for the relationships among nodes, called "edges".
   ;; The define-rule: helper procedure does this for you.
   ;; The parameters are:
   ;; WHEN: a meaningless helper reporter. Can be ommitted.
   ;; inlet-node: A string (or node variable) naming the node that may be trending up or down.
   ;;     This node feeds the rule (ede)
   ;; Relationship verb 1: A number. Can be plus 1 or minus 1. Use the helper reporter INCREASE or DECREASE
   ;; Target-node: A string (or node variable) naming the node that is affected by the trend of the source node.
   ;; Relationship verb 2: A number, just like relationship 1
   ;; BY: a meaningless helper reporter, can be savely ommitted
   ;; Relationship magnitude: A positive number. This describes the relative amount by which source affects target.
   ;;      1 is usual, and means that when source is increasing (by any amount) target increases by 1
   ;;      If a weaker relationship is desired, then use a fraction.
   ;;      If a stronger relationshiup is desired, then use a value greater than 1
   ;;      (I do not know if this is a feature of fuzzy cog maps or not, but there is it, if you want it.
   ;; A note on trend direction: By combining INCREASE and DECREASE, the trend direction is established.
   ;; For example, if both are INCREASE, or both are DECREASE, then a positive relationship is made.
   ;; (also, note that WHEN apples INCREASE oranges INCREASE is the SAME relationship as 
   ;; WHEN apples DECREASE oranges DECREASE. The one implies the other. Likewise for INCREASE and DECREASE.
   ;; However, A INCREASE B INCREASE is NOT the same rule as B INCREASE A INCREASE!
   ;; In a fuzzy-cog-map rule, only the target / outlet of the rule is changed by the rule!
      
;;   define-rule: when ill      increase suscept  decrease by 1   ;; people get sick
   define-rule: when suscept  decrease ill      increase by 1   ;; same rule ?
   define-rule: when ill      increase immune   increase by 1 ;; some people recover with immunity
   define-rule: when ill      increase suscept  increase by 1 ;; some people recover as susceptible
   define-rule: when ill      increase dead     increase by 1 ;; some sick people die
   define-rule: when immune   increase suscept  increase by 1 ;; some people lose immunity
;; define-rule: when poachers increase sheep    decrease by 1   ;; poachers steal sheep
;; define-rule: when poachers increase wolves   decrease by 0.1 ;; poachers kill wolves
;; define-rule: when wolves   increase poachers decrease by 0.5 ;; wolves kill poachers
;; define-rule: when sheps    increase poachers decrease by 0.1 ;; sheps catch poachers
;; define-rule: when sheep    increase poachers increase by 1   ;; sheep attract poachers
;; define-rule: when sheps    increase grass    increase by 1   ;; sheps expand grass

    ;; if any missing or mismated name strings occured, this will stop the model
    if errors! = true [ stop ]
   
   finalize-connections 
   setup-graphic-display
end

to create-node: [ node-name node-value node-trend lower-limit-list upper-limit-list]
   cct-nodes 1
   [ set name node-name
     set value node-value
     set trend node-trend
     ;; process limit inputs
     set has-lower-limit? first lower-limit-list
     if has-lower-limit? [ set lower-limit last lower-limit-list ]
     set has-upper-limit? first upper-limit-list
     if has-upper-limit? [ set upper-limit last upper-limit-list ]
     ;; following properties are for display purposes
     set shape "beveled-circle"
     set size 8
     set color gray + 2.5
     set label (word name " \n" value)
     set label-color black
   ]
end     

to define-rule: [ edge-inlet edge-relation-1 edge-outlet edge-relation-2 edge-mag]
   cct-edges 1
   [ 
     set inlet one-of nodes with [ name = edge-inlet ]
     set outlet one-of nodes with [ name = edge-outlet ]
     if inlet = nobody or outlet = nobody
     [ message (word "Name mismatch: looking for inlet node [ " edge-inlet " ] to match [ " edge-outlet " ]! \nCheck spelling!")
       set errors! true
       stop
     ]

     set direction edge-relation-1 * edge-relation-2
     set magnitude edge-mag
     set influence direction * magnitude
     ;; display-related properties follow
     set shape "line"
     set color black
     set label-color black
     set label influence if influence > 0 [ set label (word "+" label)] 
   ]
end
 
to finalize-connections
   ;; nodes collect the sets of edges that connect this node to other nodes
   ask nodes
   [ set inlets edges with [ outlet = myself ]
     set outlets edges with [ inlet = myself ]
   ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;
;;;;;     CREATE-NODE HELPER REPORTERS
;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;; this set of reporters provide a more human-readable syntax for the create-node entries
;; this shows an example of how a little work can improve the self-documenting features
;; of code. And since these are only used during setup, the slight extra execution time
;; added by using extra reporters, and the processing required to interpret the results
;; has no effect on the main loop execution speed
;; this is also a demonstration of one way to implement variable numnbers of inputs

to-report no-lower-limit
   report [ false ]
end

to-report no-upper-limit
   report [ false ]
end

to-report lower-limit-is [ low-value ]
   report (list true low-value)
end

to-report upper-limit-is [ high-value ]
   report (list true high-value)
end

to-report when [ node-name ]
   report node-name
end

to-report increase
   report 1
end

to-report decrease
   report -1
end

to-report by [ amount ]
   report amount
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;
;;;;;     GO 
;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Once the nodes and edges are defined, then all that's left is to iterate
;; the relationships,and possibly display the ongoing results on the screen

to go
   set iterations iterations + 1
   ifelse iterations > iteration-limit
   [ if final-text-output?
     [ go-text-output ]
     stop 
   ]
   [ get-fuzzy
     if display-intermedate-results?
     [ if text-ouput? [ go-text-output ]
       if graphic-display? [ go-graphic-display ] 
     ]
   ]
end 

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;
;;;;;     GET-FUZZY -- fuzzy cognitive map processing procedure
;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; this is the meat of the model, the procedure that performs the iterations
;; of the fuzzy cog map
  
to get-fuzzy
    
    ;; edges examine node trend, calculate net-influence of trend
    ask edges
    [ set net-influence influence * trend-of inlet ]
      
    ask nodes 
    [ ;; nodes collect influences from edges, calculate new trend
      ifelse any? inlets
      [ set trend sum values-from inlets [ net-influence ] 
        if binary-trend? [ set trend sign trend ]
      ]
      [ set trend 0 ]
               
      
      ;; calculate new value, based on influence trend      
      set old-value value
      set value value + trend
      
      ;; apply limits, if required
      
      if has-lower-limit? and value < lower-limit
      [ set value lower-limit ]
      if has-upper-limit? and value > upper-limit
      [ set value upper-limit ]
    ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;
;;;;;     DISPLAY RELATED PROCEDURES
;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; These procedures and reporters exist only to create and update the display

to setup-graphic-display
   let node-count count nodes
   let angle 360 / node-count
   ask nodes
   [ home rt angle / 3 + who * angle jump screen-edge-x * .75 ]
   ask edges
   [ 
     setxy xcor-of outlet ycor-of outlet
     ifelse is-turtle? inlet
     [ set size distance-nowrap inlet
       if size > 0
       [ set heading towards-nowrap inlet
         jump .5 * size
         rt 90
         jump 1
         lt 90
         lt 180
         set size size - 5
       ]
     ]
     [ set heading 0
       jump 5
       set heading 180
     ] 
   ]
end

to setup-text-output
end

to go-graphic-display
   ask nodes
   [ set label value
     ifelse has-lower-limit? and value = lower-limit [ set color red   + 2.5 ][
     ifelse has-upper-limit? and value = upper-limit [ set color green + 2.5 ][
            set color gray + 2.5
     ]]
     set label (word name " \n" value)
   ]
end

to go-text-output
end


to goto-turtle [ agent ]
   setxy xcor-of agent ycor-of agent
end

to goto-patch [ agent ]
   setxy pxcor-of agent pycor-of agent
end

to-report sign [ number ]
   report ifelse-value (number > 0) [ 1 ] [ ifelse-value ( number < 0 ) [ -1 ] [ 0 ] ]
end   
@#$#@#$#@
GRAPHICS-WINDOW
103
10
533
461
17
17
12.0
1
10
1
1
1
0

CC-WINDOW
5
497
801
592
Command Center

BUTTON
20
28
83
61
NIL
setup
NIL
1
T
OBSERVER
T
NIL

BUTTON
20
71
83
104
NIL
go
T
1
T
OBSERVER
T
NIL

SLIDER
561
17
733
50
iteration-limit
iteration-limit
0
500
200
10
1
NIL

SWITCH
579
88
729
121
final-text-output?
final-text-output?
0
1
-1000

SWITCH
579
129
792
162
display-intermedate-results?
display-intermedate-results?
0
1
-1000

SWITCH
578
171
697
204
text-ouput?
text-ouput?
0
1
-1000

SWITCH
579
213
727
246
graphic-display?
graphic-display?
0
1
-1000

PLOT
554
333
754
483
Plot
NIL
NIL
0.0
10.0
0.0
10.0
true
false

SWITCH
580
258
711
291
binary-trend?
binary-trend?
0
1
-1000

BUTTON
21
113
84
146
go 1
go
NIL
1
T
OBSERVER
T
NIL

BUTTON
9
170
98
203
randomize
ask nodes [ set value random 1000 + 1000 ]
NIL
1
T
OBSERVER
T
NIL

MONITOR
14
221
78
270
NIL
iterations
3
1

@#$#@#$#@
INFORMATION
--------

With this general framework, you can easily define the nodes and edges (things and relationships).

  ;; STEP ONE
   ;; Ceate local variable for each node.
   ;; The variable contains the name of the thing the node represents.
   ;; By using local variables here, we allow NetLogo to spell-check our references to the
   ;; nodes later. Whereas, if we just used the strings everywhere, misspellings can cause
   ;; frustration. Also allows one to use SHORT symbols in the code, but have LONG
   ;; descriptive names in the display and output.
   ;;
   
|   let wolves "wolves"
|   let sheep "sheep"
|   let grass "grass"
|   let sheps "shephards"
|   let poachers "poachers"
   
   ;; STEP TWO
   ;;Define the nodes.
   ;; The create-node helper procedure does this for us.
   ;; The parameters are: 
   ;; Name: A string. This is used both as a label, and as a tag used to locate and associate the nodes
   ;; Value: A number. This is the initial value of the node
   ;; Trend: A number. This is the initial trend of the node (that is, is the node currently increasing or decreasing?)
   ;; Lower-Limit: A list. This is used to define a lower limit for the nodes value, if any.
   ;;      The list may have two forms:
   ;;        If there IS a lower limit, then the list must be : [ true LIMIT ] where true is the boolean constant true, and LIMIT is a number.
   ;;        If there IS NOT a lower limit, then the list must be: [ false ]
   ;;        To make this easier and the meaning clearer in code, use the helper reporters provided.
   ;; Upper-Limit: A list. This is used to define an upper limit for the node's value, if any.
   ;;       The list is the same format as the Lower-imit.
   ;; 
   ;; The Limit helpers are:
   ;;      (no-lower-limit)
   ;;      (lower-limit-is NUMBER)
   ;;      (no-upper-limit)
   ;;      (upper-limit-is NUMBER)

|   create-node: wolves       100   1  (lower-limit-is 0) (no-upper-limit) 
|   create-node: sheep        100   1  (lower-limit-is 0) (no-upper-limit)
|   create-node: grass          1   1  (lower-limit-is 0) (no-upper-limit)
|   create-node: sheps        100   1  (lower-limit-is 0) (no-upper-limit)
|   create-node: poachers     100   1  (lower-limit-is 0) (no-upper-limit)

   ;;
   ;; Leave this line alone. If any node-name mismatches occur, this variable is changed.
   set errors! false
   
   ;; STEP THREE
   ;; Define the rules for the relationships among nodes, called "edges".
   ;; The define-rule: helper procedure does this for you.
   ;; The parameters are:
   ;; WHEN: a meaningless helper reporter. Can be ommitted.
   ;; inlet-node: A string (or node variable) naming the node that may be trending up or down.
   ;;     This node feeds the rule (edge)
   ;; Relationship verb 1: A number. Can be plus 1 or minus 1. Use the helper reporter INCREASE or DECREASE
   ;; Target-node: A string (or node variable) naming the node that is affected by the trend of the source node.
   ;; Relationship verb 2: A number, just like relationship 1
   ;; BY: a meaningless helper reporter, can be safely ommitted
   ;; Relationship magnitude: A positive number. This describes the relative amount by which source affects target.
   ;;      1 is usual, and means that when source is increasing (by any amount) target increases by 1
   ;;      If a weaker relationship is desired, then use a fraction.
   ;;      If a stronger relationshiup is desired, then use a value greater than 1
   ;;      (I do not know if this is a feature of fuzzy cog maps or not, but there is it, if you want it.
   ;; A note on trend direction: By combining INCREASE and DECREASE, the trend direction is established.
   ;; For example, if both are INCREASE, or both are DECREASE, then a positive relationship is made.
   ;; (also, note that WHEN apples INCREASE oranges INCREASE is the SAME relationship as 
   ;; WHEN apples DECREASE oranges DECREASE. The one implies the other. Likewise for INCREASE and DECREASE.
   ;; However, A INCREASE B INCREASE is NOT the same rule as B INCREASE A INCREASE!
   ;; In a fuzzy-cog-map rule, only the target / outlet of the rule is changed by the rule!
      
|   define-rule: when wolves   increase sheep    decrease by 1   ;; wolves eat sheep
|   define-rule: when sheep    increase wolves   increase by 1   ;; sheep feed wolves
|   define-rule: when grass    increase sheep    increase by 1   ;; grass feed sheep
|   define-rule: when sheep    increase sheps    increase by 1   ;; sheep employ shephards
|   define-rule: when sheps    increase wolves   increase by 1   ;; shephards kill wolves
|   define-rule: when sheps    increase sheep    increase by 1   ;; sheps husband sheep
|   define-rule: when poachers increase sheep    decrease by 1   ;; poachers steal sheep
|   define-rule: when poachers increase wolves   decrease by 0.1 ;; poachers kill wolves
|   define-rule: when wolves   increase poachers decrease by 0.5 ;; wolves kill poachers
|   define-rule: when sheps    increase poachers decrease by 0.1 ;; sheps catch poachers
|   define-rule: when sheep    increase poachers increase by 1   ;; sheep attract poachers
|   define-rule: when sheps    increase grass    increase by 1   ;; sheps expand grass


IN the interface, the only weird thing is "binary trend?"

When off, the sum of the amounts of increase or decrease of all the inlets to a node is calculated, and the total is used to determine the trend.

So, if there the inlet changes are :  +8 -2 -3 then the sum is (8 + (-2) + (-3))  = 3, so the trend is POSITIVE.

When ON, only the directions of the amounts of increase or decrease matter.
So, if the inlet changes are: +8 -2 -3, then the sum is ( 1 + (-1) + (-1)) = -1, so the trend is NEGATIVE!
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

beveled-circle
false
13
Circle -7500403 true false 0 0 300
Circle -2064490 true true 15 15 270

beveled-frame
false
13
Polygon -1 true false -90 210 -90 90 210 90 195 105 -75 105 -75 195
Rectangle -2064490 true true -75 105 195 195
Polygon -7500403 true false -90 210 -75 195 195 195 195 105 210 90 210 210

cog-16
true
0
Polygon -2674135 true false 120 150 0 150 60 135 15 90 75 105 45 45 105 75 90 15 135 60 150 0 150 120
Polygon -13345367 true false 150 120 150 0 165 60 210 15 195 75 255 45 225 105 285 90 240 135 300 150 180 150
Polygon -10899396 true false 180 150 300 150 240 165 285 210 225 195 255 255 195 225 210 285 165 240 150 300 150 180
Polygon -1184463 true false 150 180 150 300 135 240 90 285 105 225 45 255 75 195 15 210 60 165 0 150 120 150

cog-8
true
0
Polygon -2674135 true false 90 150 0 150 75 120 45 45 120 75 150 0 150 90
Polygon -13345367 true false 150 90 150 0 180 75 255 45 225 120 300 150 210 150
Polygon -10899396 true false 210 150 300 150 225 180 255 255 180 225 150 300 150 210
Polygon -1184463 true false 150 210 150 300 120 225 45 255 75 180 0 150 90 150

line
true
1
Line -2674135 true 150 0 150 300
Polygon -2674135 true true 135 45 150 0 165 45
Circle -1 true false 120 120 60
Circle -16777216 false false 120 120 60

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
