globals
[ ;; for farm color assignments
  new-color ;; the last color provided by the next-color utility
  new-color-inc ;; the direction (plus or minus) to shift the shade
  new-color-base ;; the shade of gray that is the base color for the current color series
  stop?
]


patches-own
[ farm-id
]


breed [ farms farm ]

farms-own
[ farm-root
  farm-patches
  farm-size
]

to start-up
  setup-farms
end

to reset
   ;; (for this model to work with NetLogo's new plotting features,
  ;; __clear-all-and-reset-ticks should be replaced with clear-all at
  ;; the beginning of your setup procedure and reset-ticks at the end
  ;; of the procedure.)
  ;; sanity check on mean-farm-size
  if desired-mean-farm-size < min-farm-size and seek-mean-farm-size?
  [ user-message (word "The desired mean farm size can't be less than the minimum farm size,\n"
    "or the desired mean can never be reached!\n\n"
                          "Please change either the minimum farm size or the desired mean farm size."
    )
    stop
  ]
  clear-all
  set stop? false
  reset-ticks
end

to go
end





to setup-farms
  reset
  ;; create the initial farm
  ask patch min-pxcor min-pycor
  [ sprout-farms 1
    [ set farm-size world-width
       initialize-farm
    ]
  ]
  while [ stop? != true ]
  [ build-farm
    tick
  ]
  histogram-farm-size
end

to build-farm
  ;; first, enforce MAX-FARM-SIZE, split any farms over the max size.
   ;; using ONE-OF just to slow things down in demo!
   ;; --remove for production model
   ( ifelse
    ( any? large-farms )
    [ ask one-of large-farms [ split-farm ] ]
    ( seek-mean-farm-size?)
    [ ifelse ( mean [ farm-size ] of farms > desired-mean-farm-size and any? splittable-farms )
      [ ask one-of splittable-farms [ split-farm ] ]
      [ set stop? true ]
    ]
    [ ;; otherwise
      ;; enforce MIN-FARM-COUNT
      ;; split randomly, until at least enough farms
      ifelse ( count farms < desired-min-farm-count and any? splittable-farms )
      [ ask one-of splittable-farms [ split-farm ]
      ]
      [ set stop? true ]
    ]
   )

end

to position-farm
   let scale farm-size / 2
   set size scale
   setxy ( [ pxcor ] of farm-root + scale )
         ( [ pycor ] of farm-root + scale )
end


to highlight-borders
   ask farm-patches with [ any? neighbors4 with [ farm-id != [ farm-id ] of myself ] ]
   [ set pcolor [ color + 3 ] of farm-id ]
end

to-report large-farms
   report farms with [ farm-size > max-farm-size ]
end

to-report splittable-farms
   report (farms with [ farm-size > min-farm-size ] )
end

to split-farm
   ;; do not even try to split if this farm is only 1 unit wide.
   if farm-size <= 1 [ stop ]

   ;; OK, split!
   let xx 0
   let yy 0
   let new-size farm-size / 2
   ;; create new farm-roots relative to current farm root
   ;; (do this relative to farm-root so that farm turtle's location is coupled to
   ;; determinig patches for splitting, etc)
   let new-farm-roots [ patches at-points map [ ?1 -> (list (first ?1 * new-size) (last ?1 * new-size) ) ] [ [ 0 0 ] [ 0 1 ] [ 1 1 ] [ 1 0 ] ] ] of farm-root
   ask new-farm-roots
   [ sprout-farms 1
     [ set farm-size new-size
       initialize-farm
     ]
   ]
   ;; this farm is removed
   die
end

to initialize-farm
    ;; store ID of this patch,
    ;; so we can use farm-root to id the lower-left corner of farm
    ;; letting us put farm turtle anywhere, if we =hoose.
    set farm-root patch-here
    set farm-patches patches with [ pxcor >= [ xcor ] of myself and pxcor < [ xcor + farm-size] of myself
                            and pycor >= [ ycor ] of myself and pycor < [ ycor + farm-size] of myself
                            ]
    set shape "farm"
    set color next-color
    ask farm-patches [ set pcolor [ color ] of myself set farm-id myself ]

    position-farm
end


to-report next-color
   ifelse new-color = 0 [ set new-color gray ]
   [ set new-color wrap-color (new-color + 10 ) ]
   report new-color
end

to-report between [ value minimum maximum ]
   report (minimum <= value and value <= maximum)
end

to histogram-farm-size
   set-current-plot "by farm size"
   histogram [ log farm-size 2] of farms
end
@#$#@#$#@
GRAPHICS-WINDOW
341
38
733
431
-1
-1
3.0
1
10
1
1
1
0
0
0
1
0
127
0
127
0
0
1
ticks
30.0

BUTTON
20
244
139
277
NIL
setup-farms
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
186
339
281
384
NIL
count farms
0
1
11

INPUTBOX
168
141
313
201
desired-min-farm-count
100.0
1
0
Number

INPUTBOX
127
38
222
98
min-farm-size
2.0
1
0
Number

INPUTBOX
14
38
109
98
max-farm-size
64.0
1
0
Number

PLOT
19
288
179
438
by farm size
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" ""

MONITOR
186
288
278
333
mean farm size
mean [ farm-size ] of farms
2
1
11

INPUTBOX
15
142
160
202
desired-mean-farm-size
4.0
1
0
Number

SWITCH
14
103
191
136
seek-mean-farm-size?
seek-mean-farm-size?
1
1
-1000

TEXTBOX
21
211
316
250
Turn on to seek desired mean farm size. Otherwise, divide until desired farm count is met or exceeded.
11
0.0
1

TEXTBOX
235
38
336
89
Farm sizes MUST be a power of 2: 1 2 4 8 16 32 64, etc.
11
0.0
1

TEXTBOX
15
3
459
29
STEP-WISE AREA SUBDIVISION
24
104.0
1

@#$#@#$#@
# STEP WISE AREA SUBDIVISION ```by James Steiner``` #


## WHAT IS IT?

This model demonstrates a method of generating square clusters of patches of varying sizes using progressive subdivision.

## HOW IT WORKS

A farm is a turtle that manages a square cluster of patches. By making 4 copies of the farm, and dividing the patches in the cluster among the copies, then removing the original farm, a cluster of patches is divided into four clusters.

By repeating the subdivision process, we can create clusters with the desired properties.



## HOW TO USE IT

Set the min-farm-size, max-farm-size, and min-farms to the desired values.
Note: this model does not do any sanity checking on these inputs.

min-farm-size: desired minimum number of patches that a cluster may contain. such farms can not be further subdivided.

max-in a cluster: desired maximum number of patches that a cluster may contain. such farms are always subdivided.

min-farms: the desired minimum number of farms. If the number of farms is less than this number, and there are any farms larger than the minimum size, one of them will be subdivided.

## THINGS TO NOTICE

The ratio of the count of farms of various sizes are rather consistant.

Since the selection of a farm to divide is from an even distribution, as the number of farms increases, and since these are smaller and smaller farms, it becomes more likely that the selected farm will be a small farm, and the likelyhood that a remaining large farm will remain whole increases.

## EXTENDING THE MODEL

You can use the concepts and techniques in this model to create sub-divided patch clusters for your own models.

## RELATED MODELS

See the voronoi models in the Models Library or on TurtleZero.com to see another way to subdivide patches based on a "root" patch or turtle.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

farm
false
0
Polygon -1 true false 0 300 0 135 45 60 120 60 180 60 210 30 270 30 300 60 300 300
Rectangle -7500403 true true 30 165 180 270
Rectangle -7500403 true true 210 135 270 270
Polygon -7500403 true true 30 135 60 90 150 90 180 135
Polygon -7500403 true true 210 105 210 75 225 60 255 60 270 75 270 105
Polygon -16777216 false false 0 300 0 135 45 60 180 60 210 30 270 30 300 60 300 300

plant
false
0
Polygon -1 true false 105 300 45 255 15 195 45 165 15 120 45 90 15 45 45 15 90 30 120 0 180 0 210 30 255 15 285 45 255 90 285 120 255 165 285 195 255 255 195 300
Rectangle -7500403 true true 135 90 165 285
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90
Polygon -16777216 false false 135 285 75 255 45 195 90 210 135 255 135 210 75 180 45 120 90 135 135 180 135 135 75 105 45 45 90 60 135 105 135 90 120 45 150 15 180 45 165 90 165 105 210 60 255 45 225 105 165 135 165 180 210 135 255 120 225 180 165 210 165 255 210 210 255 195 225 255 165 285

square
false
0
Rectangle -7500403 true true 0 0 300 300
@#$#@#$#@
NetLogo 6.2.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
