globals
[
  gini-index-reserve
  lorenz-points
  max-wealth
  min-wealth           ;; I included this so I could check to see if there were any negative values
  max-income
  min-income
  monthly-bi
  max-indiv-cost-of-living
  total-adults
]

patches-own
[
  patch-income      ;; the current amount of income on this patch
  monthly-income    ;; patch income / 12
]

turtles-own
[
  wealth                   ;; the amount of money a turtle has
  after-tax                ;; amount of income left after paying taxes
  min-indiv-cost-of-living ;; individual minimum cost of living based on minimum set on interface
  cost-of-living           ;; cost of living calibrated for family composition
  num-adults               ;; number of adults per household (set further down in code)
  num-children             ;; number of children per household (set further down in code)
]

;; the different family types, for setting the number of adults and children - this model leaves out roommates and extended family
breed [couples-no-children couple-no-children]
breed [couples-1-child couple-1-child]
breed [couples-2-children couple-2-children]
breed [couples-3-children couple-3-children]
breed [singles-no-children single-no-children]
breed [singles-1-child single-1-child]
breed [singles-2-children single-2-children]
breed [singles-3-children single-3-children]

;;;
;;; SETUP AND HELPERS
;;;

to setup
  clear-all
  ;; call other procedures to set up various parts of the world
  setup-patches
  setup-turtles
  update-lorenz-and-gini
  reset-ticks
end

;; set up the initial amounts of income each patch has
to setup-patches
  ask patches
  ;; set up patch income random amount depending on distribution chosen
  [
    if income-distribution = "random"
      [
        set patch-income random (mean-indiv-annual-income * 2)
      ]
    if income-distribution = "normal"
      [
        set patch-income random-normal  mean-indiv-annual-income (mean-indiv-annual-income / 2) ;; latter value is an estimate of the standard deviation based on what the curve looks like
      ]
    if income-distribution = "exponential"
      [
        set patch-income random-exponential (mean-indiv-annual-income)
      ]
      ;; keep income ≥ 0
    if patch-income < 0
      [
        set patch-income 0
      ]
    ;; set colour so richer patches have darker colour (in quintiles)
    set max-income max [ patch-income ] of patches       ;; find highest income
    if patch-income < max-income * 1 / 5 [ set pcolor 58 ]   ;; colour patches based on income levels
    if patch-income >= max-income * 1 / 5 and patch-income < max-income * 2 / 5 [ set pcolor 57 ]
    if patch-income >= max-income * 2 / 5 and patch-income < max-income * 3 / 5 [ set pcolor 56 ]
    if patch-income >= max-income * 3 / 5 and patch-income < max-income * 4 / 5 [ set pcolor 55 ]
    if patch-income >= max-income * 4 / 5 [ set pcolor 54 ]
  ]
end

;; set up the initial values for the turtle variables
to setup-turtles
  set-default-shape turtles "house"
  ;; determine composition of the different households based on demographics ( * .25 etc) entered here
  create-couples-no-children num-households * .25
  ask couples-no-children [
    set num-adults 2
    set num-children 0
  ]
  create-couples-1-child num-households * .10
  ask couples-1-child [
    set num-adults 2
    set num-children 1
  ]
  create-couples-2-children num-households * .11
  ask couples-2-children [
    set num-adults 2
    set num-children 2
  ]
  create-couples-3-children num-households * .05
  ask couples-3-children [
    set num-adults 2
    set num-children 3
  ]
  create-singles-no-children num-households * .39
  ask singles-no-children [
    set num-adults 1
    set num-children 0
  ]
  create-singles-1-child num-households * .06
  ask singles-1-child [
    set num-adults 1
    set num-children 1
  ]
  create-singles-2-children num-households * .03
  ask singles-2-children [
    set num-adults 1
    set num-children 2
  ]
  create-singles-3-children num-households * .01
  ask singles-3-children [
    set num-adults 1
    set num-children 3
  ]

  ask turtles [
    move-to one-of patches with [not any? turtles-here]
    ;; if cost of living is to vary randomly do that here, otherwise set it at the minimum cost of living
    ifelse allow-varied-indiv-min-cost-of-living [
    set min-indiv-cost-of-living random-normal min-monthly-cost-of-living-per-adult (min-monthly-cost-of-living-per-adult / 6)
      set max-indiv-cost-of-living max [min-indiv-cost-of-living] of turtles ]
    [ set min-indiv-cost-of-living min-monthly-cost-of-living-per-adult ]
    ;; adjust cost of living for family composition
    if num-adults = 1 [ set cost-of-living min-indiv-cost-of-living * ( 1 + ( num-children * top-up-children / 100 )) ]
    if num-adults = 2 [ set cost-of-living min-indiv-cost-of-living * ( 1.67 + ( num-children * top-up-children / 100 )) ]
    ;; determine wealth of individuals
    if inheritance-distribution = "random"     ;; wealth
    [
      set wealth random (mean-indiv-inheritance * 2)
    ]
    if inheritance-distribution = "normal"
    [
      set wealth random-normal mean-indiv-inheritance (mean-indiv-inheritance / 4) ;; latter value is an estimate of the standard deviation
    ]
    if inheritance-distribution = "exponential"
    [
      set wealth random-exponential (mean-indiv-inheritance)
    ]
    if wealth <= 0
    [
      set wealth 1
    ]
    ;; if there are two adults in a household, double the wealth
    if num-adults = 2 [ set wealth wealth * 2 ]
    ;; colour turtles according to their relative wealth (this is reset at every step)
    color-turtles
    ;; if there's an inheritance tax, tax wealth at the beginning during set up (if progressive, use quintiles to set tax rate)
    if inheritance-tax = "flat"
    [ set wealth wealth * ( 1 - max-tax-rate / 100 ) ]
    if inheritance-tax = "progressive"
    [ if color = 1 [ set wealth wealth * (1 - max-tax-rate / 100) ]
      if color = 2 [ set wealth wealth * (1 - 0.8 * max-tax-rate / 100) ]
      if color = 3 [ set wealth wealth * (1 - 0.6 * max-tax-rate / 100) ]
      if color = 4 [ set wealth wealth * (1 - 0.4 * max-tax-rate / 100) ]
      if color = 5 [ set wealth wealth * (1 - 0.2 * max-tax-rate / 100) ]
    ]
    ;; set the household monthly income at 1/12 of the patch income; double it for two-adult households
    set monthly-income patch-income / 12
    if num-adults = 2 [ set monthly-income monthly-income * 2 ]
    face one-of neighbors4
  ]
end

;;;
;;; GO AND HELPERS
;;;

to go
  ask turtles
  [
    upgrade-income   ;; move to neighbouring patch if it has more income
    update-wealth    ;; add net income to wealth
    color-turtles    ;; set colour according to wealth quintile
  ]
  update-lorenz-and-gini
  set min-income min [ patch-income ] of patches
  set min-wealth min [ wealth ] of turtles
  tick
end

to go-ten-years
  ask turtles
  [
    upgrade-income   ;; move to neighbouring patch if it has more income
    update-wealth    ;; add net income to wealth
    color-turtles    ;; set colour according to wealth quintile
  ]
  update-lorenz-and-gini
  tick
  if ticks = 120
  [
    reset-ticks
    stop
  ]
end


;; determine the direction which is most profitable for each turtle in
;; the surrounding patches within one patch
;; the only way I could get this to work was to have them just check one direction per tick

to upgrade-income  ;; turtle procedure - move to a higher paying patch next door if you can
  let patch-income-ahead patch-income-at-angle 0
  ;; if patch ahead has a higher income, go to it; and either way, rotate right 90°
  if (patch-income-ahead > patch-income)
  [ if not any? other turtles-on patch-ahead 1
    [ forward 1 ]]
  rt 90
  set monthly-income patch-income / 12
  if num-adults = 2 [ set monthly-income monthly-income * 2 ]
end

;; check out which neighbouring patch has the highest income (cribbed from ants)
to-report patch-income-at-angle [angle]
  let p patch-right-and-ahead angle 1
  report [patch-income] of p
end

;; add monthly income to wealth and subtract cost of living; grade cost of living to income if high enough

to update-wealth
  ;; set income tax rate: after-tax = proportion of income left after paying taxes
  if income-tax = "none" [ set after-tax 1 ]
  if income-tax = "flat" [ set after-tax 1 - max-tax-rate / 100 ]
  if income-tax = "progressive with loopholes for rich"
  [
    set max-income max [ patch-income ] of patches * 2 / 12
    if monthly-income >= max-income * 4 / 5 [ set after-tax 1 - 0.6 * (max-tax-rate / 100) ]
    if monthly-income >= max-income * 3 / 5 and monthly-income < max-income * 4 / 5 [ set after-tax 1 - 0.8 * (max-tax-rate / 100) ]
    if monthly-income >= max-income * 2 / 5 and monthly-income < max-income * 3 / 5 [ set after-tax 1 - 0.6 * (max-tax-rate / 100) ]
    if monthly-income >= max-income * 1 / 5 and monthly-income < max-income * 2 / 5 [ set after-tax 1 - 0.4 * (max-tax-rate / 100) ]
    if monthly-income < max-income * 1 / 5 [ set after-tax 1 - 0.2 * (max-tax-rate / 100) ]
  ]
  if income-tax = "progressive no loopholes for rich"
  [
    set max-income max [ patch-income ] of patches * 2 / 12
    if monthly-income >= max-income * 4 / 5 [ set after-tax 1 - 1.0 * (max-tax-rate / 100) ]
    if monthly-income >= max-income * 3 / 5 and monthly-income < max-income * 4 / 5 [ set after-tax 1 - 0.8 * (max-tax-rate / 100) ]
    if monthly-income >= max-income * 2 / 5 and monthly-income < max-income * 3 / 5 [ set after-tax 1 - 0.6 * (max-tax-rate / 100) ]
    if monthly-income >= max-income * 1 / 5 and monthly-income < max-income * 2 / 5 [ set after-tax 1 - 0.4 * (max-tax-rate / 100) ]
    if monthly-income < max-income * 1 / 5 [ set after-tax 1 - 0.2 * (max-tax-rate / 100) ]
  ]
  ;; set redistribution
  if income-redistribution = "none"
  [
    set monthly-bi 0
  ]
  if income-redistribution = "universal demogrant"   ;; give everyone the same top-up
  [
    if num-adults = 1 [ set monthly-bi max-bi-indiv / 12 ]
    if num-adults = 2
    [
      if BI-unit = "BI by individual" [ set monthly-bi 2 * max-bi-indiv / 12 ]
      if BI-unit = "BI by family" [ set monthly-bi max-bi-family / 12 ]
    ]
  ]
  if income-redistribution = "NIT with clawback"   ;; calculate partial BI after clawback
  [
    if num-adults = 1 [ set monthly-bi max-bi-indiv / 12 ]
    if num-adults = 2
    [
      if BI-unit = "BI by individual" [ set monthly-bi 2 * max-bi-indiv / 12 ]
      if BI-unit = "BI by family" [ set monthly-bi max-bi-family / 12 ]
    ]
    ifelse (monthly-income * after-tax * (clawback / 100)) < monthly-bi   ;; is after tax income too high with clawback?, if not proceed
    [
      set monthly-bi monthly-bi - (monthly-income * after-tax * (clawback / 100))
    ]
    ;; for NIT, if after-tax income is too high, don't give a BI
    [
      set monthly-bi 0
    ]
  ]
  ;; update amount of BI with amount for children
  set monthly-bi monthly-bi + monthly-bi * num-children * top-up-children / 100
  ;; calculate new wealth
  ifelse ((monthly-bi + (monthly-income * after-tax)) * 3 / 4) < cost-of-living   ;; deduct cost of living
  [set wealth wealth + monthly-bi + (monthly-income * after-tax) - cost-of-living ]
  [set wealth wealth + monthly-bi + (monthly-income * after-tax / 4) ]
end

to color-turtles
  set max-wealth  max [ wealth ] of turtles
    if wealth < max-wealth / 5
    [ set color 5 ]
    if (wealth >= max-wealth / 5 and wealth < max-wealth * 2 / 5)
    [ set color 4 ]
    if (wealth >= max-wealth * 2 / 5 and wealth < max-wealth * 3 / 5)
    [ set color 3 ]
    if (wealth >= max-wealth * 3 / 5 and wealth < max-wealth * 4 / 5)
    [ set color 2 ]
    if (wealth >= max-wealth * 4 / 5)
    [ set color 1 ]
    if wealth <= 0
    [
    set wealth 1
    set color red
    ]
end

;; this procedure recomputes the value of gini-index-reserve
;; and the points in lorenz-points for the Lorenz and Gini-Index plots
;; it is taken directly from Uri Wilensky's Wealth Distribution ABM
to update-lorenz-and-gini
    let sorted-wealths sort [wealth] of turtles
  let total-wealth sum sorted-wealths
  let wealth-sum-so-far 0
  let index 0
  set gini-index-reserve 0
  set lorenz-points []

  ;; now actually plot the Lorenz curve -- along the way, we also
  ;; calculate the Gini index.
  ;; (see the Info tab for a description of the curve and measure)
  repeat count turtles [
    set wealth-sum-so-far (wealth-sum-so-far + item index sorted-wealths)
    set lorenz-points lput ((wealth-sum-so-far / total-wealth) * 100) lorenz-points
    set index (index + 1)
    set gini-index-reserve
      gini-index-reserve +
      (index / num-households) -
      (wealth-sum-so-far / total-wealth)
  ]
end


; Copyright 1998 Uri Wilensky and 2018 Sarah Robertson
; See Info tab for full copyright and license.
@#$#@#$#@
GRAPHICS-WINDOW
483
10
937
465
-1
-1
21.24
1
10
1
1
1
0
1
1
1
-10
10
-10
10
1
1
1
ticks
30.0

BUTTON
14
52
80
85
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
91
53
172
86
go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
14
10
161
43
num-households
num-households
0
440
427.0
1
1
NIL
HORIZONTAL

SLIDER
250
151
476
184
mean-indiv-inheritance
mean-indiv-inheritance
100
1000000
46000.0
1000
1
$
HORIZONTAL

BUTTON
249
54
427
87
Go ten years (120 ticks)
go-ten-years
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
15
193
308
226
min-monthly-cost-of-living-per-adult
min-monthly-cost-of-living-per-adult
0
8000
1500.0
100
1
$
HORIZONTAL

BUTTON
179
53
242
86
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
947
261
1167
381
gini vs time
time
gini
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot (gini-index-reserve / num-households) / 0.5"

CHOOSER
157
325
334
370
income-redistribution
income-redistribution
"none" "NIT with clawback" "universal demogrant"
1

MONITOR
947
394
1055
439
gini coefficient
(gini-index-reserve / num-households) / 0.5
2
1
11

PLOT
947
10
1166
130
wealth distribution of households
Quintile
Count
0.0
6.0
0.0
10.0
true
false
"set-plot-y-range 0 num-households\n" ""
PENS
"default" 1.0 1 -16777216 true "" "plot-pen-reset\nset-plot-pen-color red\nplot count turtles with [color = red]\nset-plot-pen-color 5\nplot count turtles with [color = 5]\nset-plot-pen-color 4\nplot count turtles with [color = 4]\nset-plot-pen-color 3\nplot count turtles with [color = 3]\nset-plot-pen-color 2\nplot count turtles with [color = 2]\nset-plot-pen-color 1\nplot count turtles with [color = 1]"

CHOOSER
15
325
153
370
inheritance-tax
inheritance-tax
"none" "flat" "progressive"
2

SLIDER
339
378
472
411
clawback
clawback
0
100
44.0
1
1
%
HORIZONTAL

MONITOR
1100
393
1167
438
% in debt
100 * count turtles with [ color = red ] / num-households
2
1
11

SLIDER
266
272
438
305
max-tax-rate
max-tax-rate
0
100
50.0
1
1
%
HORIZONTAL

SLIDER
15
377
168
410
max-bi-indiv
max-bi-indiv
0
100000
22000.0
100
1
$
HORIZONTAL

PLOT
947
135
1167
255
income distribution of patches (not households)
Quintile
Count
0.0
5.0
0.0
10.0
true
false
"\n" ""
PENS
"default" 1.0 1 -16777216 true "" "plot-pen-reset\nset-plot-pen-color 58\nplot count patches with [pcolor = 58]\nset-plot-pen-color 57\nplot count patches with [pcolor = 57]\nset-plot-pen-color 56\nplot count patches with [pcolor = 56]\nset-plot-pen-color 55\nplot count patches with [pcolor = 55]\nset-plot-pen-color 54\nplot count patches with [pcolor = 54]"

CHOOSER
15
95
164
140
income-distribution
income-distribution
"random" "normal" "exponential"
1

CHOOSER
172
96
345
141
inheritance-distribution
inheritance-distribution
"random" "normal" "exponential"
2

CHOOSER
15
272
256
317
income-tax
income-tax
"none" "flat" "progressive with loopholes for rich" "progressive no loopholes for rich"
3

PLOT
1173
10
1392
130
min. cost of living distribution
Quintile
Count
0.0
5.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" "plot-pen-reset\nplot count turtles with [min-indiv-cost-of-living < max-indiv-cost-of-living / 5]\nplot count turtles with [min-indiv-cost-of-living >= max-indiv-cost-of-living / 5 and min-indiv-cost-of-living < max-indiv-cost-of-living * 2 / 5]\nplot count turtles with [min-indiv-cost-of-living >= max-indiv-cost-of-living * 2 / 5 and min-indiv-cost-of-living < max-indiv-cost-of-living * 3 / 5]\nplot count turtles with [min-indiv-cost-of-living >= max-indiv-cost-of-living * 3 / 5 and min-indiv-cost-of-living < max-indiv-cost-of-living * 4 / 5]\nplot count turtles with [min-indiv-cost-of-living >= max-indiv-cost-of-living * 4 / 5]"

SWITCH
15
232
308
265
allow-varied-indiv-min-cost-of-living
allow-varied-indiv-min-cost-of-living
1
1
-1000

CHOOSER
339
325
480
370
BI-unit
BI-unit
"BI by individual" "BI by family"
1

SLIDER
15
416
183
449
top-up-children
top-up-children
0
100
33.0
1
1
% of BI
HORIZONTAL

SLIDER
174
378
331
411
max-bi-family
max-bi-family
0
100000
31000.0
1
1
$
HORIZONTAL

TEXTBOX
178
18
429
40
Household composition entered in code tab
11
0.0
1

SLIDER
15
151
247
184
mean-indiv-annual-income
mean-indiv-annual-income
0
100000
46000.0
1
1
$
HORIZONTAL

TEXTBOX
197
426
447
444
Top up for children based on max-bi-indiv
11
0.0
1

TEXTBOX
321
193
471
235
Income and wealth doubled for families with 2 adults
11
0.0
1

@#$#@#$#@
## WHAT IS IT?

This model (and two related ones) are an attempt to see to what extent a basic income (BI), or other form of redistribution, and various forms of taxation have on the gini coefficient, which measures inequality (the higher the coefficient, the greater the economic inequality).

I wanted a way of testing various policies (e.g. taxing inheritances) to see if they made any difference to the gini coefficient relative to other policies. You can see the experiments I ran in BehaviorSpace (Tools -> BehaviorSpace). (What I found was that whether an inheritance tax makes any difference depends on the relative size of the average inheritance vs the average income.) These models were inspired by disagreements I had with others at NABIG (North American Basic Income Guarantee) Congress 2018. I don't think they're going to resolve any disagreements, though.

The model was designed for a ten-year period, with each step being one month, and ten years being 120 steps, but of course it can be run for shorter or longer periods as desired.

The gini coefficient is not going to be directly comparable to the gini coefficient in real life, since real-life economies are much more complex.

This version looks at households composed of one or two adults and 0–3 children, with no seniors or disabled people.

## HOW IT WORKS

Each agent is a household, born with a set amount of wealth (randomly assigned based on parameters you choose). If the randomly assigned amount of wealth is negative, it is set to 1. If there are two adults in the household, the initial wealth is doubled (because the coding was originally written for individuals only). This is the household's inheritance. Agent colour is set at the beginning in quintiles from grey to black based on their initial wealth (black being the wealthiest quintile), after any inheritance tax that you set has been deducted.

Each patch has a random amount of income (employment income, investment income, etc.). If a randomly assigned income level is negative, it is set to 0. The patches are shaded green in quintiles, with the highest income quintile having the darkest shade of green. This is done once during setup since patch income does not change. 

Each agent is randomly assigned to its own patch. On each step, the agent looks at the patch ahead, and moves to it if it is empty and its income is higher than the current patch. Then each agent rotates right 90° so that it faces a new direction on each step.

From there, each agent earns the monthly income for its patch, deducts its cost of living, and adds the rest to its total wealth. If the number of adults in the household is 2, the income is doubled (because the coding was originally written for individuals only). If the wealth drops below 0, the agent turns red and the wealth is reset to 1 so that the gini coefficient calculation works. For agents not in the red, agent colour is reset on every tick so that each wealth quintile has its own shade of grey, with the highest quintile being black.

The assumption is that most agents have enough income to live on (though some won't) and that they are not that mobile when it comes to work - if they see a better opening nearby they grab it but are otherwise stuck where they are. They are not perfectly capable of finding the highest paying patch nearby, but they do their best.

Note that the agents *are* perfectly rational when it comes to savings, and if they can afford it, never spend more than 75% of their income on living expenses. Obviously this is not the real world.

If there is an inheritance tax it is applied during setup. Income taxes are paid monthly along with income (one step = one month).

There is no inflation, increase in cost of living, or increase in income in the model. Nobody dies.

The gini coefficient is recalculated on each step.

## HOW TO USE IT

Choose the number of households. This could be based on the job vacancy rate, so that the number of empty patches roughly represents the number of jobs available at any one time. Set the composition of the households (number of adults, number of children) in the code tab in the setup-turtles section. You can use census data to estimate the proportion of each household type.

Choose the average (mean, not median) annual individual income (which will be divided by 12 for monthly income) and inheritance, and choose how you want these to be distributed among the population (randomly, normally, or exponentially). You can base this on real life statistics for a given country if you like, or just make something up. If you are running the model for a ten year period you should probably adjust the mean inheritance to take into account that it is only inheritances for a shorter period, rather than a lifetime. (I divided my estimated mean inheritance/wealth by about 5 to take this into account.)

Choose the lowest monthly cost of living per adult, and whether you want it to be exactly the same for each agent or distributed normally around a mean, using the allow-varied-indiv-min-cost-of-living chooser. (My experience is it doesn't make any difference to the gini coefficient, but I wanted to try it to see, so I included it and kept it in.). The household cost of living will be calculated from this value (with an additional 67% for a second adult and the top-up-children amount for each child - both of these can be changed in the code tab in the ask-turtles section), or 75% of the after-tax income, whichever is greater.

Choose what if any taxation you want to impose on agents each month, and the rate for the highest tax bracket. There are five tax brackets for progressive and semi-progressive tax rates, and they are set in the code tab.

Choose what if any income redistribution you want to give to agents each year, and set the annual amount with the max-bi slider. Decide whether or not you want to set the income redistribution on a per-individual or per-family basis with the BI-unit chooser. Also, choose how much more to provide for each child as a percentage of what a single adult would get with the top-up-children slider. (As the model is written now, this amount per child will be equal to the increased cost of living per child, but that can be changed in the code tab in the ask-turtles section.)

An NIT is a negative income tax, which gives a tax-free basic income to anyone below a certain threshold. The threshold is based on the amount of BI and the after-tax income multiplied by the clawback - if the after-clawback after-tax income is higher than the BI, the agent gets nothing. If the after-clawback after-tax income is less than the BI, the agent gets the BI plus the difference. For example, if you set the annual BI to 18,000 and have a clawback of 2/3, an annual after-tax income of 27,000 or more will not qualify someone for an NIT top-up (18,000 - (27,000 * 2/3) = 0), but an annual after-tax income of 15,000 will qualify the individual for an additional 8,000 per year (18,000 - (15,000 * 2/3)), and an after-tax income of 24,000 will qualify the individual for an additional 2,000 per year (18,000 - (24,000 * 2/3)).

A universal demogrant is a tax-free amount given to everyone regardless of how high their income is. One version of a UD is the UBI-FIT: a universal basic income plus a flat income tax (e.g. UD = 50% of median income, with everything over that taxed at a flat 50%).

If you want income redistribution to be a negative income tax (NIT) with a clawback, set the clawback. If you don't want the clawback, set it at 0%. 

(Welfare in Canada and the US is similar to a partial BI/NIT with a clawback. You can set the annual BI to be less than the annual monthly cost of living (minimum monthly cost of living times 12), if you want to make it more authentic.)

Run the scenario one step at a time, or indefinitely, or for 120 steps (the equivalent of 10 years at one step per month).

## THINGS TO NOTICE

Does the gini coefficient go in one direction over time, or does it go in one direction and then the other? Does it level off?

Even if two scenarios produce similar gini coefficients, is there any difference in the percentage of agents who go into debt (agents who are red)?

How much does each scenario outcome vary when you repeat it multiple times?

## EXTENDING THE MODEL

You can add more complex forms of taxation and/or redistribution if you want, particularly if you want to mimic a real life set of policies. You can also change the tax bracket tax rates in the code tab in the after-tax-proportion section.

You could also try to set the inheritance so that different agents receive it at different times, and it is taxed on receipt. This would probably be a good idea if you want to run it for a longer period than ten years.

There are two other models I wrote at the same time. One is just for individuals, and the other is individuals, but adding seniors and disabled people. None of them includes roommates, so you can try to set any of them so that if someone is in the red, they can try to find and link to a roommate among other agents to reduce their cost of living.

## RELATED MODELS

This model was inspired by Wealth Distribution, but other than the calculation for the gini coefficient (which I blatantly copied), it is entirely new code. There are other models listed under economics in the modelling commons that include basic income, but I don't pretend to understand them.

I have written two other companion models: Basic_Income-seniors-disabled and Basic_Income-Individual.

## CREDITS AND REFERENCES

Copyright Sarah Robertson 2019, except for the gini coefficient calculations, which are from the Wealth Distribution model by Uri Wilensky.

You can read up on the gini coefficient, or at least see why I'm using it, in the book *The Spirit Level: Why more equal societies almost always do better*, by Richard G. WIlkinson and Kate Pickett (2009).
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="indiv 18k 1.1.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 1.1.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 1.1.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 1.2.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 1.2.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 1.2.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 1.3.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 1.3.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 1.3.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 2.1.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 2.1.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 2.1.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 2.2.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 2.2.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 2.2.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 2.3.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 2.3.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 2.3.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 4.1.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 4.1.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 4.1.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 4.2.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 4.2.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 4.2.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 4.3.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 4.3.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 18k 4.3.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 1.1.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 1.1.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 1.1.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 1.2.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 1.2.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 1.2.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 1.3.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 1.3.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 1.3.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 2.1.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 2.1.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 2.1.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 2.2.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 2.2.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 2.2.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 2.3.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 2.3.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 2.3.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 4.1.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 4.1.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 4.1.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 4.2.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 4.2.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 4.2.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 4.3.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 4.3.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="indiv 22k 4.3.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by individual&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 1.1.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 1.1.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 1.1.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 1.2.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 1.2.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 1.2.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 1.3.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 1.3.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 1.3.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 2.1.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 2.1.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 2.1.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 2.2.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 2.2.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 2.2.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 2.3.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 2.3.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 2.3.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 4.1.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 4.1.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 4.1.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 4.2.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 4.2.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 4.2.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 4.3.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 4.3.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 18k 4.3.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="18000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 1.1.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 1.1.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 1.1.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 1.2.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 1.2.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 1.2.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 1.3.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 1.3.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 1.3.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 2.1.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 2.1.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 2.1.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 2.2.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 2.2.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 2.2.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 2.3.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 2.3.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 2.3.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 4.1.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 4.1.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 4.1.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 4.2.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 4.2.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 4.2.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;flat&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 4.3.1" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;none&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 4.3.2" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;universal demogrant&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="top-up-children">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-family">
      <value value="31000"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="family 22k 4.3.3" repetitions="30" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="120"/>
    <metric>(gini-index-reserve / num-households) / 0.5</metric>
    <enumeratedValueSet variable="max-tax-rate">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="min-monthly-cost-of-living-per-adult">
      <value value="1500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="clawback">
      <value value="44"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="max-bi-indiv">
      <value value="22000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-tax">
      <value value="&quot;progressive&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="BI-unit">
      <value value="&quot;BI by family&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-distribution">
      <value value="&quot;normal&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-annual-income">
      <value value="46000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-redistribution">
      <value value="&quot;NIT with clawback&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="inheritance-distribution">
      <value value="&quot;exponential&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="income-tax">
      <value value="&quot;progressive no loopholes for rich&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="allow-varied-indiv-min-cost-of-living">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="num-households">
      <value value="427"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="mean-indiv-inheritance">
      <                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      